// -*- C++ -*-
#ifndef XMLTRAVERSAL
#define XMLTRAVERSAL

#include <QHostAddress>
#include "trace.hpp"
#include "base.hpp"

/*!
 * Useful class to traverse an xml document.
 * The key idea is that the xml document is compared to a 
 * tree structure of a drive. The document node has no name
 * and is compared to the Unix like "/" folder.
 * The separator used is the '/' character. Available Unix like 
 * commands are:
 * <B>cd</B>, <B>pwd</B>, <B>mkdir</B>, <B>rmdir</B>, <B>cp</B>,
 * <B>mv</B>, <B>ls</B> and <B>exist</B>.
 *
 * QDomElement is the main node type used in this implementation, since
 * it holds attributes, which was important for our requierements.
 *
 * This class provides an iterator over dom elements of first sibling in
 * a given folder. This iterator is STL conformant, and therefor, STL 
 * algorithms can be used.
 *
 * Sorry but there's no name completion (like with the tab-tab under 
 * the unix-like bash shell), because it was too hard to implement ;-)
 *
 * <B>Example</B>:
 * <pre>
 *   <B>XMLTraversal</B> xml ;
 *   xml.<B>setDocument</B>(...) ;
 *   assert(xml.<B>is_valid</B>()) ;
 *  
 *   xml.<B>cd</B>("constants/remote/s11") ;
 *   xml.<B>cd</B>("/config/user_path") ;
 *   xml.<B>cd</B>("..") ;
 *   xml.<B>cd</B>("...") ;
 *
 *   xml.<B>rmdir</B>("/constants/halfspace/enable") ;
 *   std::cout << xml.<B>pwd</B>() << std::endl ;
 *
 *   xml.<B>cd</B>("/config/user_path") ;
 *   QDomElement e = xml.<B>currentNode</B>().toElement() ;
 *   std::cout << e.attribute("value") << std::endl ;
 *
 *   xml.<B>cd</B>("/constants/halfspace") ;
 *   if (!xml.<B>exist</B>("depth"))
 *     xml.<B>mkdir</B>("depth") ;
 *   xml.<B>mv</B>("depth", "profondeur") ;
 *
 *   QStringList list = xml.<B>ls</B>() ;
 *   for (QStringList::iterator it = list.begin(); it != list.end(); ++it) {
 *     std::cerr << (*it) << std::endl ;
 *   }
 *
 *   // Iterate over all chidren of the current directory
 *   xml.<B>cd</B>("/") ;
 *   <B>XMLTraversal::iterator</B> it = xml.<B>begin</B>() ;
 *   for (; it != xml.<B>end</B>(); it++) {
 *     std::cerr << (*it).tagName() << std::endl ;
 *   }
 * </pre>
 *
 * \author Frantz Maerten, frantz.maerten@igeoss.com (http://www.igeoss.com).  Changed by B. J. Hill
 * \version 1.02
 * \date November 2004 - 2010
 */
/**
* @brief  This class encapsulates an XML document  to make it accesible like a file system
*
* @class XMLTraversal xmltraversal.h "xmltraversal.h"
*/
class UTILITYSHARED_EXPORT XMLTraversal {
        QStack<QDomNode> savePoints;
public:
  
/**
* @brief Iterator for XMLTraversal document
*
* @class iterator xmltraversal.h "xmltraversal.h"
*/
  class UTILITYSHARED_EXPORT  iterator {
  public:
    iterator() ; // end iterator
    iterator(const QDomNode&) ; // begin iterator
    iterator(const iterator&) ;
    iterator operator ++() ;
    iterator operator ++(int) ;
    bool operator==(const iterator&) const ;
    bool operator!=(const iterator&) const ;
    QDomElement operator*() const ;
    QDomElement operator->() const ;
  private:
    iterator(const QDomNode&, const QDomNode&) ;
    QDomNode elt_ ;
    QDomNode cur_ ;
  } ;

    /**
    * @brief  Empty constructor
    *
    * @fn XMLTraversal
    */
  XMLTraversal() ;
    /**
    * @brief  Constructs an XMLTraversal object using the QString as document source
    *
    * @fn XMLTraversal
    * @param QString
    */
  XMLTraversal(QString ) ;

    /**
    * @brief Constructs an XMLTraversal object using the QDomDOcument and the document source
    *
    * @fn XMLTraversal
    * @param QDomDocument
    */
  XMLTraversal(QDomDocument) ;
    /**
    * @brief Constructs an XMLTraversal object using the QByteArray as document source
    *
    * @fn XMLTraversal
    * @param QByteArray
    */
  XMLTraversal(QByteArray );
    /**
    * @brief Copy constructor. Only copies the QDomDocument
    *
    * @fn XMLTraversal
    * @param x
    */
  XMLTraversal(XMLTraversal &x) { setDocument(x.document());}
  // =============================================
  //   Various accessors
  // =============================================

  /*!
   * Check the validity of the XMLTraversal instance
   * @return true if the internal document is valid, false otherwise
   */
  bool isValid() const ;

  /*!
   * Check if a path exists within the document
   * @return true if the path exists, false otherwise
   */
  bool exists(const QString& path) const ;

  /*!
   * Gives the current path within the XMLTraversal from the root (the 
   * document)
   * @return the current path.
   *
   * Example:
   * <pre>
   *   XMLTraversal xml ;
   *   ...// init
   *   std::cerr << xml.pwd() << std::endl ;
   * 
   *  // print: /constants/halfspace/depth
   * </pre>
   */
  QString pwd() const ;

  /*!
   * Return the node at the current path or at a specified path of the 
   * document.
   * If no path is provided, it return the node of the current path.
   * Otherwise, it returns the node of the provided path, without changing
   * the current path.
   * @param path the path to node or no argument.
   * @return the node at the current path or at the specified path, 
   * or a null node if the current path or the provided path is invalid.
   *
   * Example:
   * <pre>
   *   XMLTraversal xml ;
   *   ...// init
   *   xml.cd("constants/halfspace/depth") ;
   *   QDomNode node = xml.currentNode() ;
   *   if (node.isNull())
   *     std::cerr << "invalid current path" << std::endl ;
   *
   *   node = xml.currentNode("/constants/remote/s11") ;
   *   if (node.isNull())
   *     std::cerr << "invalid path" << std::endl ;
   * </pre>
   */
  QDomNode currentNode(const QString& path=QString()) const ;

    /**
    * @brief  returns the QDomElement of the current directory
    *
    * @fn currentElement
    * @return QDomElement
    */
  QDomElement currentElement()
  {
        if(cur_path_node_.isElement()) return cur_path_node_.toElement();
        return QDomElement();
  }

  /*!
   * Get an attribute for the current node.
   * This method assumes that the current node is a QDomElement. If not, 
   * a QString::null value is returned.
   * \return the attribute value, or QString::null if no such attribute exists,
   * or if the current node is not a QDomElement.
   */
  QString attribute(const QString& name) const ;

  /*!
   * Tells if the <b>current</b> node has the attribute attrname.
   * This method assumes that the current node is a QDomElement. If not, 
   * a false value is returned.
   * \return true if the attribute exists, or false if no such attribute 
   * exists, or if the current node is not a QDomElement.
   */
  bool hasAttribute(const QString& attrname) const ;

  /*!
   * Return the document making this XMLTraversal
   */
  const QDomDocument& document() const;

  /*!
   * Convert to string for debugging purpose
   */
  QString dump() const ;

  /*!
   * Return the list of the first child nodes of the current
   * directory. Equivalent to the Unix command ls (but with no
   * arguments).
   *
   * Example:
   * <pre>
   *   XMLTraversal xml ;
   *   ...// init
   *   QStringList list = xml.ls() ;
   *   if (list.size()==0)
   *     std::cerr << "no child nodes." << std::endl ;
   *   else {
   *     std::cerr << "Printing all child nodes of the first sibling" 
   *               << std::endl ;
   *     for (QStringList::iterator it=list.begin(); it!=list.end(); ++it) {
   *       std::cerr << (*it).latin1() << std::endl ;
   *     }
   *   }
   * </pre>
   */
  QStringList ls() const ;

  /*!
   * Iterate over the children (first sibling) of the current node of this
   * XMLTraversal. Get the first child.
   *
   * Example:
   * <pre>
   * XMLTraversal xml(mydoc) ;
   * xml.cd("/root/tag1") ;
   *
   * // Iterate over all chidren of "/root/tag1"
   * XMLTraversal::iterator it = xml.begin() ;
   * for (; it != xml.end(); it++) {
   *   std::cerr << (*it).tagName() << std::endl ;
   * }
   * </pre>
   * \see end() and size()
   */
  iterator begin() ;
  
  /*!
   * Iterate over the children (first sibling) of the current node of this
   * XMLTraversal. Get the end child (i.e. the following is true:
   * QDomElement::isNull()).
   */
  iterator end() ;

  /*!
   * Return the number of first sibling children
   */
  unsigned int size() const ;

  // =============================================
  //   Various modifiers
  // =============================================

  /*!
    * @brief  sets the content of the QDomDocument  from QString
   * The the new content of the XMLTraversal
    * @fn setDocument
   * @param doc the stringified version of the document
   * @return true if the internal document is valid, false otherwise
   */
  virtual bool setDocument(QString doc) ;
    /**
    * @brief  sets the content of the QDomDocument  from QByteArray
   * The the new content of the XMLTraversal
    * @fn setDocument
    * @param doc the stringified version of the document
    * @return bool if the internal document is valid, false otherwise
    */
  virtual bool setDocument(QByteArray doc) ;

  /*!
   * The the new content of the XMLTraversal
   * @param doc the document
   * @return true if the internal document is valid, false otherwise
   */
 virtual  bool setDocument(QDomDocument doc) ;

  /*!
   * To cd to a specific node.
   * You can use:
   *   - a usual path. If a backslash is present at the beginning of the path,
   *     it starts from the root. Otherwise, it starts from the
   *     current path (see pwd()). Examples:
   * <pre>
   *   cd("/constants/remote/s11") ;
   *   cd("../..") ;
   *   // Jump to the root:
   *   cd() ;
   *   cd("../../tools/remote/../halfspace") ;
   * </pre>
   *   - "..." to do a "cd ../.." (an alias)
   *   - no argument: to return to the root of the XMLTraversal.
   *
   * @return true if successful, false otherwise.
   */
  bool cd(const QString& path = QString()) ;
  /*!
   * To cd to a specified directory given by a QDomElement.
   * This can be useful, for example, while iterate over the children using
   * begin() and end().
   * <pre>
   *   XMLTraversal::iterator it = xml.begin() ;
   *   for (; it != xml.end(); it++) {
   *     xml.cd(*it) ;
   *     ...
   *     xml.cd("..") ;
   *   } 
   * </pre>
   */
  bool cd(const QDomElement&) ;
  // get to the root directory without having to know what it is
    /**
    * @brief Changes the working directory to the documentElement.
    *
    * @fn cdToRoot
    * @return bool returns true on success
    */
  bool cdToRoot() { return cd(QString("/%1").arg(document().documentElement().tagName()));}
  /*!
   * Create a new directory (a new node). 
   * The tail of the path is the name of the new node, and the remaining
   * path have to be a valid path. You can only create one directory at a time.
   * @return the created element. The return element is invalid (isNull()) if
   * the operation failed.
   *
   * Example:
   * <pre>
   *   XMLTraversal xml ;
   *   ...// init
   *   // Create the s23 node
   *   if (xml.mkdir("/constants/remote/s23"))
   *     std::cerr << "node successfuly created." << std::endl ;
   *   else
   *     std::cerr << "error while creating the node." << std::endl ;
   * </pre>
   */
  QDomElement mkdir(const QString& path) ;

  /*!
   * Remove an existing directory (a node)
   * @return true if succeed, false otherwise
   *
   * Example:
   * <pre>
   *   XMLTraversal xml ;
   *   ...// init
   *   if (xml.rmdir("/constants/remote"))
   *     std::cerr << "node successfuly removed." << std::endl ;
   *   else
   *     std::cerr << "error while removing the node." << std::endl ;
   * </pre>
   */
  bool rmdir(const QString& path) ;

  /*!
   * Move or rename a node. Several possible commands:
   *	 - If newpath does exist, and oldpath is not a child of newpath, 
   *	   it moves oldpath to newpath.
   *	 - If newpath doesn't exist (at least the last child of newpath),
   *	   it renames oldpath to newpath
   *
   *	@param oldpath the node to move or rename
   *	@param newpath the new parent or the new name
   *	@return true if successful, false otherwise
   *
   * Example:
   * <pre>
   *   XMLTraversal xml ;
   *   ...// init
   *   // rename remote to limits
   *   xml.mv("remote", "limits") ;
   *   // move limits to another place
   *   xml.mv("limits", "../others") ;
   * </pre>
   */
  bool mv(const QString& oldpath, const QString& newpath) ;

  /*!
   * Copy a branch node into another node.
   *	@param source the node to copy (with its children)
   *	@param target the parent for the copy node
   *	@return true if successful, false otherwise
   *
   * Example:
   * <pre>
   *   XMLTraversal xml ;
   *   ...// init
   *   // copy remote to halfspace
   *   xml.cp("/constants/remote", "/constants/halfspace") ;
   * </pre>
   */
  bool cp(const QString& source, const QString& target) ;

  /*!
   * Shortcut to set an attribute for an element.
   * \param path the path to the element
   * \param attrname the name of the attribute
   * \param attrvalue the value of the attribute.
   * \return true if successful
   */
  bool setAttribute(const QString& path,
		    const QString& attrname,
            const QString& attrvalue) ;

  /*!
   * Shortcut to set an attribute for the current element.
   * \param attrname the name of the attribute
   * \param attrvalue the value of the attribute.
   * \return true if successful
   */
  bool setAttribute(const QString& attrname, const QString& attrvalue) ;
    /**
    * @brief  Pushes the current directory onto the directory stack.
    *
    * @fn push
    */
  void push() { savePoints.push(cur_path_node_); } // push current working directory
    /**
    * @brief  Pops the top of the directory stack and makes it the current directory.
    *
    * @fn pop
    */
  void pop() { if(savePoints.count() > 0) cur_path_node_ = savePoints.pop();} // pop the CD
    /**
    * @brief  Clears the directory stack
    *
    * @fn clearStack
    */
  void clearStack() { savePoints.clear(); }
    /**
    * @brief  Adds a text node on the current element
    *
    * @fn setText
    * @param s
    */
  void setText(QString s); // set text on current element
    /**
    * @brief Return the text node for  the element at the given path.
    *
    * @fn text
    * @param path the directory to get the text from
    * @return QString
    */
  QString text(QString path) // get the text for a node - restore current after fetch
  {
   QString res("");
    push();
    if(cd(path)) res = currentElement().text();
    pop();
   return res;
  }
    /**
    * @brief Creates the directory and change to directory
    *
    * @fn mkdirAndCd
    * @param s path to create
    * @return QDomElement the element of the new directory
    */
  QDomElement mkdirAndCd(QString s) // create and change
  {
    QDomElement e = mkdir(s);
    cd(s);
    return e;
  };

    /**
    * @brief Creates the directory, changes to the directory and sets the text node of the element.
    *
    * @fn mkdirSetText
    * @param directory
    * @param text
    * @return QDomElement the new directory
    */
  QDomElement mkdirSetText(QString directory, QString text) // create and change
  {
    QDomElement e = mkdir(directory);
    cd(e);
    setText(text);
    return e;
  };

    /**
    * @brief  Loads a document from the given file
    *
    * @fn loadFromFile
    * @param s file name
    * @return bool returns true if successful
    */
 bool loadFromFile(QString s)
 {
     QFile f(s);
     if(f.open(QIODevice::ReadOnly))
     {
        doc_.clear();
        markUpdated();
        return doc_.setContent(&f);
     };
     return false;
 }

 /**
  * @brief Saves the XML to file as text
  * Returns true on success
  * @param s   File name
  * @return bool
 */
 bool saveToFile(QString s)
 {
     QFile f(s);
     if(f.open(QIODevice::WriteOnly))
     {
         QTextStream os(&f);
         doc_.documentElement().setAttribute("Updated",updateTime.toString(Qt::ISODate));
         os << doc_;
         return true;
     }
     return false;
 }
// get and set a variant value
    /**
    * @brief  Encodes the variant , creates a sub directory and sets the text to the QVariant::toString
    *The XmlGet is the inverse of the XmlSet. This makes writing dialog level Getters and Setters easier as one is just the copy of the other , changing only
    * G to S.
    * @fn XmlSet
    * @param s   Subdirectory name
    * @param v   QVariant to write to XML
    */
void XmlSet ( QString s, QVariant v)
{
     push();
     mkdir(s);
     cd(s);
    VariantToElement(v,currentElement());
    pop();
};

    /**
    * @brief  Converts the named sub-element's text to QVariant
    *
    * @fn XmlGet
    * @param s  Subdirectory to load from
    * @param b QVariant to write to
    */
void  XmlGet ( QString s, QVariant &b)
 {
    push();
    if(cd(s))
    {
        b = (elementToVariant(currentElement()));
    }
    pop();
};


    /**
    * @brief Writes a QHostAddress object to the given subdirectory
    *
    * @fn XmlSet
    * @param s  Subdirectory to write to
    * @param a  QHostAddress  object to write
    */
void XmlSet ( QString s, QHostAddress a)
{
     push();
     mkdir(s);
     cd(s);
    VariantToElement(QVariant(a.toString()),currentElement());
    pop();
};

    /**
    * @brief  Reads a QHostAddress object from the given subdirectory
    *
    * @fn XmlGet
    * @param s  Subdirectory to read from
    * @param b  QHostAddress to read into
    */
void  XmlGet ( QString s, QHostAddress &b)
 {
    push();
    if(cd(s))
    {
        QVariant v = (elementToVariant(currentElement()));
        b = QHostAddress(v.toString());
    }
    pop();
};


    /**
    * @brief  Reads a QVariant from the given subdirectory
    *
    * @fn GetVar
    * @param s subdirectory to read from
    * @return QVariant
    */
QVariant GetVar ( QString s)
 {
    QVariant b;
    XmlGet(s,b);
    return b;
};

//
// these all assume the pwd is at the root  of the config tree
// is each control has a sub-directory the same as the object name
//
/**
  @brief The following are convience functions to get/set QWidget based control values to / from XML documents.
The values are saved under the item name that is the same as the control object name.

  @list
void XmlGet(QLineEdit *item)
void XmlGet(QLabel *item)
void XmlSet(QLineEdit *item)
void XmlSet(QLabel *item)
void XmlGet(QDoubleSpinBox *item)
void XmlSet(QDoubleSpinBox *item)
void XmlGet(QSpinBox *item)
void XmlSet(QSpinBox *item)
void XmlGet(QComboBox *item)
void XmlSet(QComboBox *item)
void XmlGetText(QComboBox *item)
void XmlSetText(QComboBox *item)
void XmlGet(QAbstractButton *item)
void XmlSet(QAbstractButton *item)
void XmlGet(QDateTimeEdit *item)
void XmlSet(QDateTimeEdit *item)
void XmlGet(QListWidget *item, bool fGetChecked = false)
void XmlSet(QListWidget *item, bool fSetChecked = false)
void XmlGet(QPlainTextEdit *item)
void XmlSet(QPlainTextEdit *item)

  @endlist


  */
#ifdef QT_GUI_LIB

void XmlGet(QLineEdit *item){ item->setText(GetVar(item->objectName()).toString());}
void XmlGet(QLabel *item){ item->setText(GetVar(item->objectName()).toString());}
void XmlSet(QLineEdit *item) {XmlSet(item->objectName(),QVariant(item->text()));}
void XmlSet(QLabel *item) {XmlSet(item->objectName(),QVariant(item->text()));}
void XmlGet(QDoubleSpinBox *item){ item->setValue(GetVar(item->objectName()).toDouble());}
void XmlSet(QDoubleSpinBox *item) {XmlSet(item->objectName(),QVariant(item->value()));}
void XmlGet(QSpinBox *item){ item->setValue(GetVar(item->objectName()).toInt());}
void XmlSet(QSpinBox *item) {XmlSet(item->objectName(),QVariant(item->value()));}
void XmlGet(QComboBox *item){ item->setCurrentIndex(GetVar(item->objectName()).toInt());}
void XmlSet(QComboBox *item) {XmlSet(item->objectName(),QVariant(item->currentIndex()));}
void XmlGetText(QComboBox *item)
{
    setComboText(item,GetVar(item->objectName()).toString());
}
void XmlSetText(QComboBox *item) {XmlSet(item->objectName(),QVariant(item->currentText()));}
//
// this covers most buttons
//
void XmlGet(QAbstractButton *item){ item->setChecked(GetVar(item->objectName()).toBool());}
void XmlSet(QAbstractButton *item) {XmlSet(item->objectName(),QVariant(item->isChecked()));}
//

void XmlGet(QDateTimeEdit *item){ item->setDateTime(GetVar(item->objectName()).toDateTime());}
void XmlSet(QDateTimeEdit *item) {XmlSet(item->objectName(),QVariant(item->dateTime()));}

// this gets and sets lists . if Get/Set Checked is true then the checked state of the listed items is handled
void XmlGet(QListWidget *item, bool fGetChecked = false)
{
    QStringList l = GetVar(item->objectName()).toStringList();
    if(fGetChecked)
    {
        // assumes the list has been populated
        setCheckList(item, l);
    }
    else
    {
        item->clear();
        item->addItems(l);
    }
}
void XmlSet(QListWidget *item, bool fSetChecked = false)
{
    QStringList l;
    if(fSetChecked)
    {
      l = getCheckList(item); // only save the checked item list
    }
    else
    {
        for(int i = 0; i < item->count(); i++) l << item->item(i)->text(); // save the text of all the items
    }
    XmlSet(item->objectName(),QVariant(l));
}
void XmlGet(QPlainTextEdit *item) { item->setPlainText(GetVar(item->objectName()).toString()); }
void XmlSet(QPlainTextEdit *item) {XmlSet(item->objectName(),QVariant(item->toPlainText()));}
#endif
//
    /**
    * @brief  Initialises the document and sets the root directory to the given name
    *
    * @fn initialise
    * @param s  root directory name
    */
void initialise(QString s = "Document")
{
   doc_.clear();
    clearStack();
    cur_path_node_ = doc_;
    doc_.appendChild (doc_.createElement(s)); // create the document element
    markUpdated();
}

    /**
    * @brief  Converts the text at the given element to a QVariant
    *
    * @fn elementToVariant
    * @param e Element to read
    * @return QVariant
    */
 QVariant elementToVariant(QDomElement e);

    /**
    * @brief  Stores a QVariant at the given QDomElement
    *The Variant is usually converted to text. However QMap and QVariantArray  and QStringList are encoded and sub elements
    * @fn VariantToElement
    * @param v
    * @param e
    */
 void VariantToElement(QVariant v, QDomElement e);

    /**
    * @brief Returns the time of the last update
    *
    * @fn getUpdateTime
    * @return QDateTime
    */
 QDateTime getUpdateTime() const { return updateTime;}
    /**
    * @brief  Sets the update time to the current date time
    *
    * @fn QDateTime::currentDateTime
    */
 void markUpdated() { updateTime = QDateTime::currentDateTime();}
private:
  QDomNode cd_to_node(const QString& path) ;
  QDomDocument doc_ ;
  QDomNode cur_path_node_ ;
  QDateTime updateTime; // when last updated

} ;

// helper macros - can only use in class member function

template <class T>
    /**
    * @brief  Templated variant value set
    *
    * @fn XmlSet
    * @param x  XMLTraversal document to write to
    * @param s    Subdirectory to write value to
    * @param <T>b Value to write
    */
XMLTraversal & XmlSet (XMLTraversal &x, QString s, T b)
{
    QVariant v;
    v.setValue<T>(b);
     x.XmlSet(s,v);
     return x;
};
    /**
    * @brief  Templated variant value get
    *
    * @fn XmlGet
    * @param x  XMLTraversal document to write to
    * @param s    Subdirectory to write value to
    * @param <T>b Value to  read
    */
template <class T>
        XMLTraversal &  XmlGet (XMLTraversal &x, QString s, T &b)
 {
    QVariant v;
    x.XmlGet(s,v);
    b = qVariantValue<T>(v);
    return x;
};


// sets the variable name as the element tagname and then sets the text for the element to the value
    /**
    * @brief  Macro to save a variable in an XMLTraversal document under an item name the same as the variable name
    *@fn XMLSET
    * @param x  XMLTraversal to write to
    * @param v  Variable to write
    */
#define XMLSET(x,v)  { XmlSet(x,#v,v);}
// inverse of set
/**
* @brief  Macro to read a variable from an XMLTraversal document under an item name the same as the variable name
*@fn XMLGET
* @param x  XMLTraversal to read from
* @param v  Variable to read
*/

#define XMLGET(x,v) { XmlGet(x,#v,v);}

inline  XMLTraversal & XmlSet (XMLTraversal &x, QString s, QVariant v)
{
    x.XmlSet(s,v);
    return x;
};

inline XMLTraversal &  XmlGet (XMLTraversal &x, QString s, QVariant &b)
 {
    x.XmlGet(s,b);
    return x;
};

// sets the variable name as the elemetn tagname and then sets the text for the element to the value
#define XMLSETVAR(x,v)  { XmlSetVar(x,#v,v);}
// inverse of set
#define XMLGETVAR(x,v) { XmlGetVar(x,#v,v);}


// -----------------------------------------------------------------

inline const QDomDocument & XMLTraversal::document() const  {
  return doc_ ;
}

inline XMLTraversal::iterator XMLTraversal::begin() {
  return iterator(currentNode()) ;
}

inline XMLTraversal::iterator XMLTraversal::end() {
  return iterator() ;
}

inline unsigned int XMLTraversal::size() const {
  int i = 0 ;
  QDomNode node = cur_path_node_.firstChild() ;
  while(!node.isNull()) {
    if (node.isElement())
      i++ ;
    node = node.nextSibling() ;
  }
  return i ;
}

inline bool XMLTraversal::isValid() const {
  return (!doc_.isNull()) ;
}

inline QString XMLTraversal::attribute(const QString& name) const {
  if (!cur_path_node_.isElement())
    return QString::null ;
  return (const_cast<QDomNode&>(cur_path_node_)).
    toElement().attribute(name) ;
}

inline bool XMLTraversal::hasAttribute(const QString& attrname) const {
  if (!cur_path_node_.isElement())
    return false ;
  return (const_cast<QDomNode&>(cur_path_node_)).
    toElement().hasAttribute(attrname) ;
}

inline bool XMLTraversal::exists(const QString& d) const {
  XMLTraversal* This = const_cast<XMLTraversal*>(this) ;
  QDomNode cur = cur_path_node_ ;
  QDomNode node = This->cd_to_node(d) ;
  This->cur_path_node_ = cur ;
  return !node.isNull() ;
}

inline QString XMLTraversal::dump() const {
  return doc_.toString() ;
}



inline bool XMLTraversal::iterator::operator!=(const iterator& i) const {
  return !(operator==(i)) ;
}

inline QDomElement XMLTraversal::iterator::operator*() const {
  return const_cast<QDomNode&>(cur_).toElement() ;
}

inline QDomElement XMLTraversal::iterator::operator->() const {
  return const_cast<QDomNode&>(cur_).toElement() ;
}

inline bool XMLTraversal::iterator::operator==(const iterator& i) const {
  if (i.cur_.isNull() && cur_.isNull())
    return true ;

  if (i.elt_ == elt_ && i.cur_ == cur_)
    return true ;
  return false ;
}



#endif

