#ifndef STATS_HPP
#define STATS_HPP
/*!
 * \file stats.hpp
 *
 * \author B. J. Hill
 * \date __DATE__
 *  License:  GNU LESSER GENERAL PUBLIC LICENSE 2.1
 *  (c)  Micro Research Limited 2010 -
 *  $Id$
 */

#include "base.hpp"
#include "xmltraversal.h"



/**
* @brief  General statistics class for populations with convenience features for data acquisition systems
This class calculates the statisics of values added to it. This class includes SPC features. Upper and lower control limits can be defined and are tested each time a new value is added to the
object. The times of data addition and when control limits are exceeded are tracked.
* @class Statistics stats.hpp "stats.hpp"
*/
    class UTILITYSHARED_EXPORT Statistics
	{
		double lastValue;
		unsigned numberSamples;
		double sum;
		double sumSquares;
		double minimum;
		double maximum;
		//
		// SPC metrics
		//
		bool trackSpc;
		double upperControl;
		double lowerControl;
		//
		unsigned trendCount;
		unsigned meanCrowding;
                unsigned triggerCount;
		bool dirTrendUp;
		bool dirTrendDown;
		//
                QDateTime updateTime;
                bool upperControlEnabled;
                bool lowerControlEnabled;
                QDateTime lowerControlTriggerTime;
                QDateTime upperControlTriggerTime;
                //
		public:
    /**
    * @brief Constructs a Statistics object
    *
    * @fn Statistics
    */
                Statistics() : lastValue(0),numberSamples(0),sum(0),
		sumSquares(0),minimum(0),maximum(0),
		trackSpc(false),upperControl(0),lowerControl(0),
		trendCount(0),meanCrowding(0),
                triggerCount(0),dirTrendUp(false),
                dirTrendDown(false),
                upperControlEnabled(false),
                lowerControlEnabled(false)
		{
		};

    /**
    * @brief  Copy constructor
    *
    * @fn Statistics
    * @param s Object to copy
    */
                Statistics(const Statistics &s) : lastValue(s.lastValue),numberSamples(s.numberSamples),sum(s.sum),
		sumSquares(s.sumSquares),minimum(s.minimum),maximum(s.maximum),
		trackSpc(s.trackSpc),upperControl(s.upperControl),lowerControl(s.lowerControl),
                trendCount(s.trendCount),meanCrowding(s.meanCrowding),triggerCount(s.triggerCount),
                dirTrendUp(false),dirTrendDown(false),updateTime(s.updateTime),
                upperControlEnabled(s.upperControlEnabled),
                lowerControlEnabled(s.lowerControlEnabled),
                lowerControlTriggerTime(s.lowerControlTriggerTime),
                upperControlTriggerTime(s.upperControlTriggerTime)
		{
		};
    /**
    * @brief  reset the statistics tracks
    *
    * @fn clear
    */
                virtual void clear()
		{
			lastValue = sum = sumSquares = minimum = maximum = 0;
                        numberSamples = trendCount = meanCrowding = triggerCount = 0;
			dirTrendUp = dirTrendDown = false;
		};
    /**
    * @brief adds a new value and updates the statistics
    *
    * @fn setValue
    * @param v value to add to statistics
    */
                virtual void setValue(double v);
    /**
    * @brief Gets the last value added to the statisics object
    *
    * @fn getLastValue
    * @return double
    */
                double getLastValue() const { return lastValue;};
    /**
    * @brief Get the number of samples added to the set
    *
    * @fn getNumberSamples
    * @return unsigned
    */
                unsigned getNumberSamples() const { return numberSamples;};
    /**
    * @brief Returns the minimum of the values added to the statitics
    *
    * @fn getMinimum
    * @return double
    */
                double getMinimum() const { return minimum;};
    /**
    * @brief Returns the maximum of the values added to the statisics
    *
    * @fn getMaximum
    * @return double
    */
                double getMaximum() const { return maximum;};
    /**
    * @brief Returns maximum minus minimum
    *
    * @fn getRange
    * @return double
    */
                double getRange() const { return  maximum - minimum;};
    /**
    * @brief Returns the T-Value for the statisics set
    *
    * @fn tval
    * @param pLevel
    * @param degreesOfFreedom
    * @return double
    */
                static double tval(double pLevel, int degreesOfFreedom);
    /**
    * @brief  Returns the sum of values added
    *
    * @fn getSum
    * @return double
    */
                double getSum() const { return sum;};
    /**
    * @brief Returns the upper control  limit
    *
    * @fn getUpperControl
    * @return double
    */
                double getUpperControl() const { return upperControl;}
    /**
    * @brief Returns the lower control limit
    *
    * @fn getLowerControl
    * @return double
    */
                double getLowerControl() const {return lowerControl;}
    /**
    * @brief Returns if the upper control limit is enabled
    *
    * @fn getUpperControlEnabled
    * @return bool
    */
                bool getUpperControlEnabled() const { return upperControlEnabled;}
    /**
    * @brief  Returns if the lower control limit is enabled
    *
    * @fn getLowerControlEnabled
    * @return bool
    */
                bool getLowerControlEnabled() const { return lowerControlEnabled;}
    /**
    * @brief Sets the enable state for the upper control limit
    *
    * @fn setUpperControlEnabled
    * @param f  Set to true to enable control limit
    */
                void setUpperControlEnabled(bool f) { upperControlEnabled = f;}
    /**
    * @brief Sets the enable state for the lower control limit
    *
    * @fn setLowerControlEnabled
    * @param f Set to true to enable the control limit
    */
                void setLowerControlEnabled(bool f) { lowerControlEnabled = f;}
    /**
    * @brief Gets the time of the last value added to the statisics
    *
    * @fn getUpdateTime
    * @return QDateTime
    */
                QDateTime getUpdateTime() const { return updateTime;}
    /**
    * @brief  Returns the last time the upper control limit was exceeded, if enabled
    *
    * @fn getUpperControlTriggerTime
    * @return QDateTime
    */
                QDateTime getUpperControlTriggerTime() const { return upperControlTriggerTime;}
    /**
    * @brief Returns the time the lower control limit was exceeded, if enabled
    *
    * @fn getLowerControlTriggerTime
    * @return QDateTime
    */
                QDateTime getLowerControlTriggerTime() const { return lowerControlTriggerTime;}

    /**
    * @brief  Returns the variance of the values
    *
    * @fn variance
    * @return double
    */
                double variance() const
		{
			if ( getNumberSamples() > 1)
			{
				return(( sumSquares - ((sum * sum) /  getNumberSamples())) / ( getNumberSamples() - 1));
			}
			else
			{
				return ( 0.0 );
			}
		};
		// standard deviation
    /**
    * @brief  Returns the standard deviation of the values
    *
    * @fn getStdDev
    * @return double
    */
                double getStdDev() const
		{
			if ( getNumberSamples() <= 0 || variance() <= 0)
			{
				return(0);
			}
			else
			{
				return( (double) sqrt( variance() ) );
			}
		};
		//
    /**
    * @brief  The determines the confidence value such that the given percentage of values lie within mean +/- the confidence value
    *
    * @fn confidence
    * @param interval  This is the percentage confidence to be evaluated
    * @return double
    */
                double confidence(int interval) const
		{
                    double p = ((double)interval) / 100.0;
                    return confidence(p);
		};
		//
    /**
    * @brief  The determines the confidence value such that the given fraction of values lie within mean +/- the confidence value
    *
    * @fn confidence
    * @param p_value fraction of values to calculate confidence limit for
    * @return double
    */
                double	confidence(double p_value) const
		{
			int df = getNumberSamples() - 1;
			if (df <= 0) return HUGE_VAL;
			double t = tval((1.0 + p_value) * 0.5, df);
			if (t == HUGE_VAL)
			return t;
			else
			return (t * getStdDev()) / sqrt(double(getNumberSamples()));
		}
    /**
    * @brief  Returns the mean of the values
    *
    * @fn getMean
    * @return double
    */
                double getMean() const
		{
			if(numberSamples > 0)
			{
				return sum / (double)numberSamples;
			};
			return 0;
		};
		//
    /**
    * @brief Returns the number of consecitively increasing or decreasing values that have been added.
    This values can indicate a drift in the population. This is an SPC metric.
    *
    * @fn getTrendCount
    * @return unsigned
    */
                unsigned getTrendCount() const { return trendCount;};
    /**
    * @brief Returns the number of consectiuve values that have exceed the control limits.
    This is an SPC metric. Typically the control limits are set at one standard deviation. Therefore one in three values should exceed the control limits.
    Typically three consecutive values outside the control limits is an indicator of a fault condition
    *
    * @fn getTriggerCount
    * @return unsigned
    */
                unsigned getTriggerCount() const { return triggerCount;};
    /**
    * @brief Returns the numbe rof consecutive values not outside the control limits
    *This is an SPC metric. As one in three values should lie outside the control limits the absence of values outside the control limits
    can indicate a fault. Typically a mean crowding value of  > 10 is taken to mean either the control limits are incorrect or the data being measured is faulty
    (eg failed sensor))
    * @fn getMeanCrowding
    * @return unsigned
    */
                unsigned getMeanCrowding() const { return meanCrowding;};
    /**
    * @brief  Returrns true if SPC tracking is enabled
    *
    * @fn getTrackSpc
    * @return bool
    */
                bool getTrackSpc() const { return trackSpc;};
    /**
    * @brief  Sets or clears SPC tracking
    *
    * @fn setTrackSpc
    * @param f  True enables SPC tracking
    */
                void setTrackSpc(bool f) { trackSpc = f;};
    /**
    * @brief Writes the statistics to and XML document
    *
    * @fn toXML
    * @param x XML document to write to
    * @return XMLTraversal
    */
                XMLTraversal & toXML(XMLTraversal &x)
                {
                    x.push();
                    XMLSET(x,lastValue);
                    XMLSET(x,numberSamples);
                    XMLSET(x,sum);
                    XMLSET(x,sumSquares);
                    XMLSET(x,minimum);
                    XMLSET(x,maximum);
                    //
                    // SPC metrics
                    //
                    XMLSET(x,trackSpc);
                    XMLSET(x,upperControl);
                    XMLSET(x,lowerControl);
                    XMLSET(x,upperControlEnabled);
                    XMLSET(x,lowerControlEnabled);
                    //
                    XMLSET(x,trendCount);
                    XMLSET(x,meanCrowding);
                    XMLSET(x,triggerCount);
                    XMLSET(x,dirTrendUp);
                    XMLSET(x,dirTrendDown);
                    //
                    XMLSET(x,updateTime);
                    XMLSET(x,upperControlTriggerTime);
                    XMLSET(x,lowerControlTriggerTime);

                    x.pop();
                    return x;
                };

    /**
    * @brief  Reads the statistics object from an XML document
    *
    * @fn fromXML
    * @param x XML document to  read from
    * @return XMLTraversal
    */
                XMLTraversal & fromXML(XMLTraversal &x)
                {
                    x.push();
                    // determine stats from this
                    XMLGET(x,lastValue);
                    XMLGET(x,numberSamples);
                    XMLGET(x,sum);
                    XMLGET(x,sumSquares);
                    XMLGET(x,minimum);
                    XMLGET(x,maximum);
                    //
                    // SPC metrics
                    //
                    XMLGET(x,trackSpc);
                    XMLGET(x,upperControl);
                    XMLGET(x,lowerControl);
                    XMLGET(x,upperControlEnabled);
                    XMLGET(x,lowerControlEnabled);
                    //
                    XMLGET(x,trendCount);
                    XMLGET(x,meanCrowding);
                    XMLGET(x,triggerCount);
                    XMLGET(x,dirTrendUp);
                    XMLGET(x,dirTrendDown);
                    XMLGET(x,updateTime);
                   XMLGET(x,upperControlTriggerTime);
                   XMLGET(x,lowerControlTriggerTime);
                    x.pop();
                    return x;
                };

                //
		friend QDataStream & operator >> (QDataStream &is, Statistics &r);
		friend QDataStream & operator << (QDataStream &os, const Statistics &r);
	};
	//
    /**
    * @brief  QDatastream input operator
    *
    * @fn operator >>
    * @param is Input stream
    * @param r   Object to read to
    */
        inline QDataStream & operator >> (QDataStream &is, Statistics &r)
	{
		is >> r.lastValue;
		is >> r.numberSamples;
		is >> r.sum;
		is >> r.sumSquares;
		is >> r.minimum;
		is >> r.maximum;
		//
		// SPC metrics
		//
		is >> r.trackSpc;
		is >> r.upperControl;
		is >> r.lowerControl;
		//
		is >> r.trendCount;
		is >> r.meanCrowding;
                is >> r.triggerCount;
		is >> r.dirTrendUp;
		is >> r.dirTrendDown;
                is >> r.updateTime;
                is >> r.upperControlTriggerTime;
                is >> r.lowerControlTriggerTime;
		return is;
	};
	//
    /**
    * @brief  QDatastream output operator
    *
    * @fn operator <<
    * @param is   output stream
    * @param r object to write
    */
        inline QDataStream & operator << (QDataStream &is, const Statistics &r)
	{
		is << r.lastValue;
		is << r.numberSamples;
		is << r.sum;
		is << r.sumSquares;
		is << r.minimum;
		is << r.maximum;
		//
		// SPC metrics
		//
		is << r.trackSpc;
		is << r.upperControl;
		is << r.lowerControl;
		//
		is << r.trendCount;
		is << r.meanCrowding;
                is << r.triggerCount;
		is << r.dirTrendUp;
		is << r.dirTrendDown;
                is << r.updateTime;
                is << r.upperControlTriggerTime;
                is << r.lowerControlTriggerTime;
                return is;
	};



        // used for testing a value against
/**
* @brief This class handles SCADA threshold triggers.
SCADA systems typically have four action / event limits; HiHi, HiLo,LoHi,LoLo.
Some times called upper alarm, upper warning, lower warning aand lower alarm.
This class  encapsulates one threshold.
*
* @class StatisticsThreshold stats.hpp "stats.hpp"
*/
        class  StatisticsThreshold
        {
            double _threshold;
            int          _triggerCount;
            bool      _enabled;
            bool     _triggered;
        public:

            typedef enum
            {
                None = 0,
                HiHi,
                HiLo,
                LoHi,
                LoLo,
                NumberThresholds
            } ThresholdTypes;


    /**
    * @brief This constructs a threshold
    *
    * @fn StatisticsThreshold
    * @param t threshold limit
    */
            StatisticsThreshold(double t = 0) :
                    _threshold(t),
                    _triggerCount(0),
                    _enabled(false),
                    _triggered(false) {}
    /**
    * @brief Copy constructor
    *
    * @fn StatisticsThreshold
    * @param s object to copy from
    */
            StatisticsThreshold( const StatisticsThreshold &s):
                    _threshold(s._threshold),
                    _triggerCount(s._triggerCount),
                    _enabled(s._enabled),
                    _triggered(s._triggered){}
            //
    /**
    * @brief returns the trigger threshold
    *
    * @fn threshold
    * @return double
    */
            double threshold() const { return _threshold;}
    /**
    * @brief sets the trigger threshold
    *
    * @fn setThreshold
    * @param t  the threshold
    * @param e threshold enable
    */
            void setThreshold(double t, bool e = true) { _threshold = t; _enabled = e;}
    /**
    * @brief returns the number of times the threshold has been triggered
    *
    * @fn triggerCount
    * @return int
    */
            int triggerCount() const { return _triggerCount;}
    /**
    * @brief resets the trigger count to zero and clears the triggered flag
    *
    * @fn clear
    */
            void clear() { _triggerCount = 0; _triggered = false;}
            //
    /**
    * @brief  compares the value with the treshold and triggers if it more than the threshold
    *Returns true if triggered.
    * @fn compareHi
    * @param v
    * @return bool
    */
            bool compareHi(double v)
            {
                _triggered = ( v  >= _threshold) && _enabled;
                if( _triggered)_triggerCount++;
                return _triggered;
            }
            //
    /**
    * @brief  compares the value with the treshold and triggers if it less than the threshold
    *Returns true if triggered
    * @fn compareLo
    * @param v value to compare
    * @return bool
    */
            bool compareLo(double v)
            {
                _triggered = ( v <= _threshold) && _enabled;
                if(_triggered)_triggerCount++;
                return _triggered;
            }
            //
    /**
    * @brief  increments the triggere count
    *
    * @fn increment
    */
            void increment() { _triggerCount++;}
    /**
    * @brief  returns true if the threshold is enabled
    *
    * @fn enabled
    * @return bool
    */
            bool enabled(){ return _enabled;}
    /**
    * @brief  sets the enabled state of the trigger
    *
    * @fn setEnabled
    * @param f if true the threshold is enabled
    */
            void setEnabled(bool f) { _enabled = f;}
    /**
    * @brief returns the trigger state of the trheshold
    *
    * @fn triggered
    * @return bool
    */
            bool triggered() const { return _triggered;}
            //
        };

        // statistics with alarm thresholds
/**
* @brief  This class is a Statistics class with the four standard SCADA thresholds, HiHi, HiLo, LoHi,LoLo
* As each value is
* @class StatisticsThresholdSet stats.hpp "stats.hpp"
*/
        class StatisticsThresholdSet : public Statistics
        {
            QVector<StatisticsThreshold> _thresholds;
            //
            bool _triggered; // any of thresholds triggered after setValue
            bool _hihilolo; // hihi or lolo triggered
            bool _hilolohi; // hilo or lohi triggered
            //
        public:
    /**
    * @brief Constructs
    *
    * @fn StatisticsThresholdSet
    */
            StatisticsThresholdSet() :
                    _thresholds(StatisticsThreshold::NumberThresholds),
                    _triggered(false),_hihilolo(false),_hilolohi(false) {}
            //
    /**
    * @brief  Copy constructor
    *
    * @fn StatisticsThresholdSet
    * @param s object to copy
    */
            StatisticsThresholdSet(const StatisticsThresholdSet &s) :
             Statistics(s),_thresholds(s._thresholds),
             _triggered(s._triggered),
             _hihilolo(s._hihilolo),
             _hilolohi(s._hilolohi)
            {

            }
            //
    /**
    * @brief returns the indexed threshold
    *
    * @fn thresholds
    * @param i threshold index
    * @return StatisticsThreshold
    */
            StatisticsThreshold & thresholds(int i) { return _thresholds[i];}
            //
    /**
    * @brief  sets all thresholds
    *
    * @fn setThresholds
    * @param lolo LoLo threshold
    * @param lohi  LoHi threshold
    * @param hilo HiLo threshold
    * @param hihi  HiHi threshold
    * @param loloEnable LoLo enable
    * @param lohiEnable LoHi enable
    * @param hiloEnable HiLo enable
    * @param hihiEnable HiHi enable
    */
            void setThresholds(double lolo, double lohi, double hilo, double hihi,
                               bool loloEnable = true, bool lohiEnable = true, bool hiloEnable = true, bool hihiEnable = true)
            {
                _thresholds[StatisticsThreshold::HiHi].setThreshold(hihi,hihiEnable);
                _thresholds[StatisticsThreshold::HiLo].setThreshold(hilo,hiloEnable);
                _thresholds[StatisticsThreshold::LoHi].setThreshold(lohi,lohiEnable);
                _thresholds[StatisticsThreshold::LoLo].setThreshold(lolo,loloEnable);
            }
            //
    /**
    * @brief  returns true if any threshold has been triggered by the last value added
    *
    * @fn triggered
    * @return bool
    */
            bool triggered() const { return _triggered;} // any levels triggered
    /**
    * @brief  returns true if either HiHi or LoLo has been triggered
    *
    * @fn triggeredHiHiLoLo
    * @return bool
    */
            bool triggeredHiHiLoLo() const { return _hihilolo;}
    /**
    * @brief returns true if either HiLo or LoHi have been triggered.
    *
    * @fn triggeredHiLoLoHi
    * @return bool
    */
            bool triggeredHiLoLoHi() const { return _hilolohi;}
            //
            // return the highest state
            StatisticsThreshold::ThresholdTypes maxState()
            {
                if(triggered())
                {
                    if(_thresholds[StatisticsThreshold::HiHi].triggered()) return StatisticsThreshold::HiHi;
                    if(_thresholds[StatisticsThreshold::LoLo].triggered()) return StatisticsThreshold::LoLo;
                    if(_thresholds[StatisticsThreshold::HiLo].triggered()) return StatisticsThreshold::HiLo;
                    if(_thresholds[StatisticsThreshold::LoHi].triggered()) return StatisticsThreshold::LoHi;
                }
                return StatisticsThreshold::None;
            }

            //
    /**
    * @brief Adds a new value to the statistics. The thersholds are tested and triggered.
    *
    * @fn setValue
    * @param v
    */
            void setValue(double v)
            {
                _hihilolo =  _thresholds[StatisticsThreshold::LoLo].compareLo(v) || _thresholds[StatisticsThreshold::HiHi].compareHi(v);
                _hilolohi =  _thresholds[StatisticsThreshold::HiLo].compareHi(v) ||  _thresholds[StatisticsThreshold::LoHi].compareLo(v);
                _triggered = _hihilolo || _hilolohi;
                if(!triggered()) _thresholds[StatisticsThreshold::None].increment(); // nothing changed
                Statistics::setValue(v);
            }
            //
    /**
    * @brief Clears and resets the statistics object and trigger states
    *
    * @fn clear
    */
            void clear()
            {
                for(int i = 0; i < StatisticsThreshold::NumberThresholds; i++) _thresholds[i].clear();
                Statistics::clear();
                _triggered =    _hihilolo =     _hilolohi = false;
            }
            //
        };
#endif
