#include "snsspread.h"
#include "snsconstants.h"
/*!
 * \file snsspread.cpp
 *
 * \author B. J. Hill
 * \date __DATE__
 *  License:  GNU LESSER GENERAL PUBLIC LICENSE 2.1
 *  (c)  Micro Research Limited 2010 -
 *  $Id$
 */

/**
 * @brief Constructs a Spread connection object. Call connect to attach to the server
 *
 * @param parent object
 * @
*/
SnsSpread::SnsSpread(QObject *parent) :
    QObject(parent),_mailbox(0),_num_groups(0)
{

}

/**
 * @brief Destroy a spread connection
 *
 * @return
*/
SnsSpread::~SnsSpread()
{
    disconnectFrom();
}



/**
 * @brief
 *
 * @param spread_host Host to connect to
 * @param spread_port Port to connect on
 * @param private_name The connection name
 * @param priority The priority of the connection
 * @param group_membership If to belong to a group
 * @param timeOut Timeout for connecting. Default indefinite.
 * @return result of connection attempt
*/
int SnsSpread::connectTo(QString spread_host, int spread_port,
                       QString private_name, int priority, bool group_membership, Timeout timeOut)

{
    //
    char name[SNS_MAX_NAME_SIZE];
    qsnprintf ( name, SNS_MAX_NAME_SIZE, "%d@%s", spread_port,(const char *)spread_host.toAscii() );
    //
    DBG("Name " << name);
    //
    char private_group[MAX_PRIVATE_NAME + 2];
    //
    memset(private_group,0,sizeof(private_group));
    //
    int ret =  ILLEGAL_SPREAD;
    QByteArray pn = private_name.toAscii();
    //
    if(timeOut.isZero())
    {
        DBG("Zero Timeout")

        ret = SP_connect (name,
                           private_name.isEmpty()?0:pn.data(),
                           priority,
                           group_membership?1:0,
                           &_mailbox,
                           private_group);
    }
    else
    {
        ret = SP_connect_timeout (name,
                           private_name.isEmpty()?0:pn.data(),
                           priority,
                           group_membership?1:0,
                           &_mailbox,
                           private_group,
                           timeOut);
    }
   // _privateName = QByteArray(private_group);
    if(ret != ACCEPT_SESSION)
    {
        _mailbox = 0;
        _num_groups = 0;
    }
    else
    {
        _num_groups = 1; // always have the private group
    }
    GDS << "ret " << ret << " Mailbox " << (int)_mailbox << "Spread Host " << spread_host << " port " << spread_port << " private name " << private_name;
    return ret;
}

/**
 * @brief does an SP_disconnect if  the mailbox is valid.
 *
 * @return result of SP_disconnect
*/
int SnsSpread::disconnectFrom()
{
    int ret = 0;
    if(isConnected())
    {
        ret = SP_disconnect(_mailbox);
    }
    _mailbox = 0;
    _num_groups = 0;
    DBG("ret = " << ret)
    return ret;
}

/**
 * @brief Join a group on a spread server
 *
 * @return int
*/
int SnsSpread::join( QString group)
{
    int ret = -1;
    if(isConnected())
    {
        QByteArray g = group.toAscii();
        ret = SP_join(_mailbox,g.data());
        if(!ret )_num_groups++;
    }
    return ret;
}

/**
 * @brief Leave a spread group
 *
 * @return int
*/
int SnsSpread::leave(QString group)
{
    int ret = -1;
    if(isConnected())
    {
        QByteArray g = group.toAscii();
        ret = SP_leave(_mailbox,g.data());
        if(!ret && (_num_groups > 0))_num_groups--;
    }
    return ret;
}

/**
 * @brief
 *
 * @return int
*/
int SnsSpread::multicast(service service_type, QString group, int mess_type, QByteArray b)
{
    int ret = -1;
    if(isConnected())
    {
        SP_multicast (_mailbox, service_type, (const char *)group.toAscii(), mess_type, b.size(), b.data());
    }
    return ret;
}

/**
 * @brief Receives a message from a spread server
 *
 * @param m Message reference to receive message
 * @param groupList List of groups message was sent to
 * @return size of message
*/
int SnsSpread::receive(SnsSpread::Message &m, QStringList &groupList)
{
    int ret = -1;
    DBG("_num_groups " << _num_groups)
    if(isConnected() && (_num_groups > 0))
    {
        char sender[MAX_GROUP_NAME];
        sender[0] = 0;
        int num_groups = 0;
        char groups[256][MAX_GROUP_NAME];
        //
        m._data.resize(SNS_MESSAGE_SIZE);
        m._service_type = 0;
        ret =  SP_receive (_mailbox, &m._service_type, sender,
                    256, &num_groups, groups,
                    &m._mess_type, &m._endian_mismatch,
                    m._data.size(), m._data.data());
        if(ret >= 0)
        {
            m._sender = QString(sender);
            groupList.clear();
            for(int i = 0; i < num_groups; i++)
            {
                groupList << QString(groups[i]);
            }
            m._data.resize(ret); // resize the byte array to suit

        }
        DBG("Ret= " << ret << " Actual number of groups " << num_groups << " Groups " << groupList.join(","))
    }
    return ret;
}

/**
 * @brief Test to see if two ids are the same
 *
 * @return int
*/
int SnsSpread::equalGroupIds(group_id g1, group_id g2)
{
    return SP_equal_group_ids(g1,g2);
}


/**
 * @brief Polls a connection for messages. Call before calling receive (which blocks)
 *
 * @return The number of bytes pending
*/
int SnsSpread::poll()
{
    int ret = -1;
    if(isConnected())
    {
        ret = SP_poll(_mailbox);
    }
    return ret;
}

/**
 * @brief Prints last error to standard out/error
 *
*/
void SnsSpread::error(int e)
{
    SP_error(e);
}

/**
 * @brief Kills a connection without calling disconnect
 *
*/
void SnsSpread::kill()
{
    if(isConnected())
    {
        SP_kill(_mailbox);
        _num_groups = 0;
    }
    _mailbox = 0;
}
