#ifndef ROLLINGBUFFER_HPP
#define ROLLINGBUFFER_HPP
/*!
 * \file rollingbuffer.hpp
 *
 * \author B. J. Hill
 * \date __DATE__
 *  License:  GNU LESSER GENERAL PUBLIC LICENSE 2.1
 *  (c)  Micro Research Limited 2010 -
 *  $Id$
 */
	//
#include "base.hpp"
#include "stats.hpp"
#include "xmltraversal.h"
	//
/**
* @brief  Rolling buffers store the last n values and  evaluates the statisics on the buffered values.
Typically these are used in SPC type applications.
*
* @class RollingBuffer rollingbuffer.hpp "rollingbuffer.hpp"
*/
    class UTILITYSHARED_EXPORT RollingBuffer
	{
		int Width;
		//
		QList<double> buffer;
		//
		bool changed; // if true stats need to be recalculated
                StatisticsThresholdSet stats; // the current statistic of the buffer
		//
		public:
    /**
    * @brief  Constructs a rolling buffer  of given size.
    *
    * @fn RollingBuffer
    * @param width Buffer Size
    */
                RollingBuffer( int width = 8)  : Width(width){}
    /**
    * @brief  Copy constructor
    *
    * @fn RollingBuffer
    * @param r  Object to copy
    */
                RollingBuffer(const RollingBuffer &r)  :
                        Width(r.Width),buffer(r.buffer),
                        changed(r.changed),stats(r.stats){}

    /**
    * @brief  Clears buffer and resets the statisics
    *
    * @fn clearBuffer
    */
                void clearBuffer()
		{
			stats.clear();
			buffer.clear();
		};
		//
    /**
    * @brief Gets the buffer width
    *
    * @fn getWidth
    * @return int
    */
                int getWidth() const { return Width;}
		//
    /**
    * @brief  Sets the buffer width
    *
    * @fn setWidth
    * @param width
    */
                void setWidth(int width)
		{
                    if(width > 0)
                    {
                        Width = width;
                        if(buffer.count() > width)
			{
                                while(buffer.count() > width) buffer.pop_front();
				changed = true;
                        }
                     }
                }

    /**
    * @brief Gets the current statistics without re-evaluating the statistics of the buffered values
    *
    * @fn readStatistics
    * @return StatisticsThresholdSet
    */
                StatisticsThresholdSet & readStatistics() { return stats;}
    /**
    * @brief Evaluates the statistics of the buffer and returns the result.
    *
    * @fn getStatistics
    * @return StatisticsThresholdSet
    */
                StatisticsThresholdSet & getStatistics()
		{
			if(changed)
			{
				// recalculate if necessary
				stats.clear();
				for(int i = 0; i < buffer.count(); i++)
				{
					stats.setValue(buffer.value(i));
				};
			};
			changed = false;
			return stats;
		};
                // this does not invert - maybe change for persistance
    /**
    * @brief  Writes the rolling buffer statistics to XML, after evaluating statistics.
    *
    * @fn toXML
    * @param x  The XMLTraversal object to write to.
    * @return XMLTraversal
    */
                XMLTraversal & toXML(XMLTraversal &x)
		{
			getStatistics();
                        return stats.toXML(x);
		};

    /**
    * @brief  The statistics are read from the XML document
    *
    * @fn fromXML
    * @param x The XMLTraversal object to read from
    * @return XMLTraversal
    */
                XMLTraversal & fromXML(XMLTraversal &x)
                {
                        return stats.fromXML(x);
                };

    /**
    * @brief  Adds a value to the rolling buffer. If more than width values are in the buffer then the oldest value is dropped.
    *
    * @fn addValue
    * @param v  Value to add to buffer
    */
                void addValue(double v)
		{
			changed = true;
			buffer.push_back(v);
			if(buffer.count() > Width)
			{
				while(buffer.count() > Width) buffer.pop_front();
			};
		};
		friend QDataStream & operator >> (QDataStream &is, RollingBuffer &r);
		friend QDataStream & operator << (QDataStream &os, const RollingBuffer &r);
	};
    /**
    * @brief  QDatastream input operator
    *
    * @fn operator >>
    * @param is  Stream to read from
    * @param r object to read into
    */
        inline QDataStream & operator >> (QDataStream &is, RollingBuffer &r)
	{
		int w;
		is >> w;
		r.setWidth(w);
		r.buffer.clear();
		is >> r.buffer;
		r.changed = true;
		return is;
	};
    /**
    * @brief  QDatastream output operator
    *
    * @fn operator <<
    * @param os Stream to write to
    * @param r object to write
    */
        inline QDataStream & operator << (QDataStream &os, const RollingBuffer &r)
	{
		os << r.getWidth();
		os << r.buffer;
		return os;
	};
#endif
