#ifndef LOGGING_H
#define LOGGING_H
#include <QxtCore>

#include <QList>
#include <QVariant>
/*!
 * \file [filename]
 *
 * \author B. J. Hill
 * \date      __DATE__
 *  License:  GNU LESSER GENERAL PUBLIC LICENSE 2.1
 *  (c)  Micro Research Limited 2010 -
 *  $Id$
 */


#include <QSqlQuery>
#include <QDateTime>
#include <QxtLoggerEngine>

// macros for logging
// positional marker for logging
#define LM QString("%1[%2][%3]").arg(__FILE__).arg(__FUNCTION__).arg(__LINE__)
#define LOGT QList<QVariant> () << LM

/*!
 * \brief Initialises Qxt logging engine
 *
 * The Qxt logging engine is initialised to its default state. Fatal, Critical, Error and Warning messages are logged. Trace and Debug messages are disabled.
 *
 * \param in logFileName
 * \return  none
 */

inline void initialiseLogging(QString logFileName)
{
    qxtLog-> addLoggerEngine ( "FileLog" , new QxtBasicFileLoggerEngine (QString("%1.log").arg(logFileName)));
    qxtLog->initLoggerEngine("FileLog");
    //
    qxtLog-> enableLogLevels ( "FileLog" , QxtLogger::FatalLevel );
    qxtLog-> enableLogLevels ( "FileLog" , QxtLogger::CriticalLevel  );
    qxtLog-> enableLogLevels ( "FileLog" , QxtLogger::ErrorLevel );
    qxtLog-> enableLogLevels ( "FileLog" , QxtLogger::WarningLevel  );
    qxtLog-> disableLogLevels ( "FileLog" , QxtLogger::TraceLevel );
    qxtLog-> disableLogLevels ( "FileLog" , QxtLogger::DebugLevel);
    //
}

// define special logs for debug and trace to file
#define LOG_DEC(s) \
qxtLog-> addLoggerEngine ( #s"Log" , new QxtBasicFileLoggerEngine (QString("%1/%2.log").arg(f).arg(#s))); \
        qxtLog->initLoggerEngine(#s"Log");

#ifdef QT_SQL_LIB
#include <QSqlDatabase>
/*!
 * \class DatabaseLogger
 *
 * \brief  This class write Qxt logging messages to a database table
 *
 * This is a QxtLoggerEngine class that interfaces between QxtLogger and a database. Messages are stored in a tabel with the following fields:
LOG_TIMESTAMP (time stamp), LOG_LEVEL (the Qxt logging level) and LOG_MESSAGE (the message text)
 *
 * \author B. J. Hill
 * \date
 */
class DatabaseLogger : public QxtLoggerEngine
{
    QString _dbname;
    QString _loggerName;
    QString _tableName;
public:


    /**
    * @brief  Constructs a Database Logger object
    *
    * @fn DatabaseLogger
    * @param databaseName  QSqlDatabase to use - must already exist
    * @param logFileName The Qxt Logger name.
    * @param tableName The database table to create and write to.
    */
    DatabaseLogger(QString databaseName, QString logFileName, QString tableName) :
            _dbname(databaseName),_loggerName(logFileName),_tableName(tableName)
    {

    }
    /**
    * @brief  Creates the table to write messages to.
    *
    * @fn createTable
    */
    virtual void createTable()
    {
        QSqlDatabase db = QSqlDatabase::database(_dbname,true);
        db.exec(
        QString(
          "CREATE TABLE %1 IFNOTEXIST (LOG_TIMESTAMP DATETIME DEFAULT CURRENT_TIMESTAMP,"
          "LOG_LEVEL CHAR(16) DEFAULT 'InfoLevel',"
          "LOG_MESSAGE VARCHAR(512)"
          ");"
          ).arg(_tableName));
    }


    /**
    * @brief This initialises the logger engine
    *
    * @fn initLoggerEngine
    */
    virtual void    initLoggerEngine(){}

    /**
    * @brief  This termiantes the logger engine
    *
    * @fn killLoggerEngine
    */
    virtual void    killLoggerEngine(){}
    /**
    * @brief  Returns true to indicate the engine is initialised.
    *
    * @fn isInitialized
    * @return bool
    */
    virtual bool    isInitialized() const { return true;}

    /**
    * @brief  Returns the name of the database being used.
    *
    * @fn dbname
    * @return QString
    */
    QString dbname() const {return _dbname;}

    /**
    * @brief  Returns the logger name
    *
    * @fn loggerName
    * @return QString
    */
    QString loggerName() const { return _loggerName;}

    /**
    * @brief Formats and writes the log message to the database.
    *
    * @fn writeFormatted
    * @param level The Qxt logging level
    * @param msgs  The list of items to be formatted into a message
    */
    virtual void    writeFormatted(QxtLogger::LogLevel level, const QList<QVariant>& msgs);
};
#endif

/**
* @brief  This class allows a logger engine to signal a QObject with a log message
*
* @class LoggerLink Logging.h "Logging.h"
*/
class LoggerLink : public QObject
{
    Q_OBJECT
public:
    /**
    * @brief  Constructs the LoggerLink
    *
    * @fn LoggerLink
    * @param parent The parent QObject
    */
    LoggerLink(QObject *parent  = 0) : QObject(parent){};
public slots:
    /**
    * @brief  This slot receives messages and posts them on.
    *
    * @fn logOutput
    * @param level Qxt logger level
    * @param msg The log message
    */
    void logOutput(QString level, QString msg) {emit log(level,msg);}
signals:
    /**
    * @brief  This signals a message on from a logging engine
    *
    * @fn log
    * @param level Qxt logger level
    * @param msg The log message
    */
     void log(QString level, QString msg);
};

/**
* @brief This class is a Qxt Logger Engine that signals messages on to other QObjects
*
* @class GuiLogger Logging.h "Logging.h"
*/
class GuiLogger : public QxtLoggerEngine
{
    LoggerLink _link;
public:
    /**
    * @brief Constructs the logging engine
    *
    * @fn GuiLogger
    */
    GuiLogger(){}
    /**
    * @brief  Initialises the logging engine
    *
    * @fn initLoggerEngine
    */
    virtual void    initLoggerEngine(){};
    /**
    * @brief  Kills the logging engine
    *
    * @fn killLoggerEngine
    */
    virtual void    killLoggerEngine(){} ;
    /**
    * @brief  Returns true to indicate the engine has been initialised
    *
    * @fn isInitialized
    * @return bool
    */
    virtual bool    isInitialized() const { return true;}
    /**
    * @brief  Formats the logging messages and uses a LoggerLink to forward them on.
    *
    * @fn writeFormatted
    * @param level Qxt Logging level
    * @param msgs The set of items to be formatted
    */
    virtual void    writeFormatted(QxtLogger::LogLevel level, const QList<QVariant>& msgs)
    {
        QStringList l;
        l << QDateTime::currentDateTime().toString(Qt::ISODate);
        for(int i = 0; i < msgs.count(); ++i) l << msgs[i].toString();
        _link.logOutput(QxtLogger::logLevelToString ( level ),l.join(" "));
    }
    /**
    * @brief  Returns the link object
    *
    * @fn link
    * @return const LoggerLink
    */
    const LoggerLink * link()  const { return &_link;}
};

#endif // LOGGING_H
