#ifndef SERVERRESOURCES_H
#define SERVERRESOURCES_H
//
// this class encapsulates the server resources that may be shared amongst tasks
//
// the resources are:
//
// serial ports
// databases
// networks
//
#include <Utility/base.hpp>
#include <Utility/xmltraversal.h>
#include "MrlHardware_global.h"
//
// this is the definition of a configured serial port
//
struct PortSettings;
class MRLHARDWARESHARED_EXPORT SerialPortResource
{
public:
    typedef enum
    {
        Serial = 0 , // attached to machine
        Network // a network adapter
    } SerialPortType;

    // serial parity
    typedef enum
    {
            NoParity = 0,
            Odd,
            Even
    } Parity;

    // network protocol
    typedef enum
    {
        Tcp = 0,
        Udp
    } NetworkProtocol;

private:

    SerialPortType _type;
    //
    //
// device name (COM1 or ttyS* or ttyUSB*, under linux etc /dev is implicit)
	QString _device;
// baud rate

        union
        {
            unsigned _speed;
            int _networkPort;
        };
// number of data bits
        union
        {
            unsigned _data_bits;
            NetworkProtocol _protocol;
        };
// number of stop bits
	unsigned _stop_bits;
// the partity
        Parity _parity;

	public:
                SerialPortResource() : _type(Serial) {}
                SerialPortResource(SerialPortType t, QString d, unsigned s = 9600, unsigned db = 8, unsigned sb = 1, Parity p = NoParity)
                        : _type(t),_device(d), _speed(s), _data_bits(db),_stop_bits(sb),_parity(p)
                {
                }
// copy
		SerialPortResource(const SerialPortResource &s)
                        : _type(s._type),
                        _device(s._device),
                         _speed(s._speed),
                         _data_bits(s._data_bits),
                        _stop_bits(s._stop_bits),
                        _parity(s._parity){}

// accessors
		QString device() const { return _device;}
                SerialPortType type() { return _type;}
                QString networkAddress() const { return _device;}
                int networkPort() const { return _networkPort;}
		unsigned speed() const { return _speed;}
		unsigned dataBits() const { return _data_bits;}
		unsigned stopBits() const { return _stop_bits;}
                Parity parity() const { return _parity;}
                NetworkProtocol protocol() const { return _protocol;}
//
		void setDevice(QString s) { _device = s;}
                void setType(SerialPortType t) {_type = t;}
                void setNetworkAddress(QString s) { _device = s;}
		void setSpeed(unsigned s) { _speed = s;}
                void setNetworkPort(int s) {_networkPort = s;}
		void setDataBits(unsigned s) {_data_bits = s;}
		void setStopBits(unsigned s) {_stop_bits = s;}
                void setPartity(Parity s) {_parity = s;}
                void setProtocol(NetworkProtocol s) { _protocol = s;}
                //
                void toPortSettings(PortSettings &ps);
//
		XMLTraversal & toXML(XMLTraversal &x)
		{
//
// write to an XMLTraversal document
// serial and network union fields
			x.push();
                        XmlSet(x,"Device",_device); // device or host address
                        XmlSet(x,"Type",(int)_type); // the serial port type
                        XmlSet(x,"Speed",_speed); // also network port
                        XmlSet(x,"NetworkPort",_speed); // also network port
                        XmlSet(x,"DataBits",_data_bits); // also netowrk protocol if type = 1
			XmlSet(x,"StopBits",_stop_bits);
                        XmlSet(x,"Parity", (int)_parity);
//
			x.pop();
//
			return x;
		}
//
		XMLTraversal & fromXML(XMLTraversal &x)
		{
//
// write to an XMLTraversal document
//
			x.push();
                        int t = 0;
                        XmlGet(x,"Device",_device);
                        XmlGet(x,"Type",t);
                        _type = (SerialPortType)t;
                        XmlGet(x,"Speed",_speed);
                        XmlGet(x,"DataBits",_data_bits);
                        XmlGet(x,"StopBits",_stop_bits);
                        XmlGet(x,"Parity",t);
                        _parity = (Parity)t;
//
			x.pop();
//
			return x;
		}
};

//
class  SerialPortResourceMap : public QMap<QString,SerialPortResource>
{
	public:
		SerialPortResourceMap(){}
		XMLTraversal & toXML(XMLTraversal &x);
		XMLTraversal & fromXML(XMLTraversal &x);
};




// definition of a database on the server
class  ServerDatabase
{
// database name under Qt - database objects are in a global table
	QString _name;
// database type
	QString _type;
// database host
	QString _host;
// database name on host
	QString _db;
// user name
	QString _user;
// password
	QString _pw;
	public:
		ServerDatabase(QString name = "database", QString type = "QMYSQL3", QString host="localhost" , QString db = "data", QString user = "", QString pw = "")
			: _name(name), _type(type), _host(host), _db(db), _user(user), _pw(pw) {}

		ServerDatabase(const ServerDatabase &s)
			: _name(s._name), _type(s._type), _host(s._host), _db(s._db), _user(s._user), _pw(s._pw) {}
//
// accessors
//
		QString name() const { return _name;}
		QString type() const { return _type;}
		QString host() const { return _host;}
		QString db() const { return _db;}
		QString user() const { return _user;}
		QString pw() const { return _pw;}
//
		void setName(QString s) { _name = s;}
		void setHost(QString s) { _host = s;}
		void setDb(QString s) { _db = s;}
		void setUser(QString s) { _user = s;}
		void setPw(QString s) {_pw =s;}
		void setType(QString s) {_type =s;}
//
                // create and open the databsase
                QSqlDatabase  createOpen(QString serverName = "")
                {
                    QSqlDatabase db;
                    // on the server side we do not need the server qualification
                    QString cn = serverName.isEmpty()?_name:QString("%1%2").arg(serverName).arg(_name);
                    if(!QSqlDatabase::contains(cn))
                    {
                       db = QSqlDatabase::addDatabase(_type,cn);
                       db.setDatabaseName(_db);
                       db.setHostName(_host);
                       db.setUserName(_user);
                       db.setPassword(_pw);
                    }
                    else
                    {
                        db = QSqlDatabase::database(cn);
                    }
                    if(!db.isOpen()) db.open();
                    return db;
                }

//
		XMLTraversal & toXML(XMLTraversal &x)
		{
			x.push();
			x.mkdirAndCd(_name);
			XmlSet(x,"Type",_type);
			XmlSet(x,"Host",_host);
			XmlSet(x,"Db",_db);
			XmlSet(x,"User",_user);
			XmlSet(x,"Pw",_pw);
			x.pop();
			return x;
		}
		XMLTraversal & fromXML(XMLTraversal &x)
		{
			x.push();
			if(x.cd(_name))
			{
				XmlGet(x,"Type",_type);
				XmlGet(x,"Host",_host);
				XmlGet(x,"Db",_db);
				XmlGet(x,"User",_user);
				XmlGet(x,"Pw",_pw);
			}
			x.pop();
			return x;
		}

};

class  ServerDatabaseMap : public QMap<QString,ServerDatabase>
{
	public:
		ServerDatabaseMap() {}
		XMLTraversal & toXML(XMLTraversal &x);
		XMLTraversal & fromXML(XMLTraversal &x);
};

// what is available on this server
class  ServerResources
{
//
// all ports available on the system
	QStringList availableSerialPorts;
// types of database supported
	QStringList databaseTypes;
// database configuration on the server
	ServerDatabaseMap databases;

//
	public:
		ServerResources();
		XMLTraversal & toXML(XMLTraversal &x);
		XMLTraversal & fromXML(XMLTraversal &x);
// find what the computer system supports
		void detect()
		{
			availableSerialPorts = detectSerialPorts();
			databaseTypes = detectDatabaseTypes();
			
		}
//
// accessors - the data is only config data not the implimentation
//
		QStringList getAvailableSerialPorts() const { return availableSerialPorts;}
		QStringList getDatabaseTypes() const { return databaseTypes;}
		ServerDatabaseMap & getDatabases() { return databases;}
		
//
// detect which serial ports are present
		static QStringList detectSerialPorts();
// get the available database types
		static QStringList detectDatabaseTypes();

};

// SERVERRESOURCES_H
#endif
