#ifndef PACKETDRIVER_H
#define PACKETDRIVER_H
/*!
 * \file packetdriver.h
 *
 * \author B. J. Hill
 * \date __DATE__
 *  License:  GNU LESSER GENERAL PUBLIC LICENSE 2.1
 *  (c)  Micro Research Limited 2010 -
 *  $Id$
 */
//
#include <Utility/base.hpp>
//
#include "MrlHardware_global.h"
#define PACKET_DRIVER_MAXRETRY (2)
// stop things getting too big
#define PACKET_DRIVER_MAX_QUEUE_LEN (1000)
/*!
 * \class PacketDriver
 *
 * \brief This class handles sending/receiving message packets.
 *
 * Usually communication to external instrumentation or systems is done by a send/response
 protocol where the messages have well defined start and end markers.

 Derived classes support sending messages over serial, TCP and UDP.
 *
 * \author B. J. Hill
 * \date
 */class MRLHARDWARESHARED_EXPORT PacketDriver : public QObject
{
    Q_OBJECT
    QMutex _mux; // locking mutex

    static QMap<QString,PacketDriver *> _drivers; // set of packet drivers
    QMap<unsigned, QObject *> _callers;
    /**
     * @brief Packet request structure
     *
    */
    struct Packet
    {
        unsigned _caller; // the caller id
        unsigned _request; // the request id
        QByteArray _packet; // the data
        unsigned _retry;
        Packet(unsigned c, unsigned r, QByteArray p) :
                _caller(c),_request(r),_packet(p),_retry(0){}
        Packet(const Packet &p) :
                _caller(p._caller),
                _request(p._request),
                _packet(p._packet),
                _retry(p._retry)
                {}
    };
protected:
    QQueue<Packet> _queue; // request queue
    QBuffer _rx; // received packet
private:
    //
    bool _packetStart; // found the packet start
    unsigned _state; // state machine
    bool _waiting; // waiting for requests
    //
    QIODevice *_chan;// I/O channel
    int _timeout;        // timeout period
    QTime _timer;      // timeout timer
    //
    char _startChar; // the start character
    char _endChar; // the endcharacter
    //
    bool _skipWhiteSpace; // true if leading white space is skipped
public:
    /**
     * @brief Constructs and registers the packet driver
     *
     * @param n Packet driver name
     * @param parent Parent QObject
    */
    explicit PacketDriver(QString n, QObject *parent = 0);
    /**
     * @brief Destructor
     *
    */
    ~PacketDriver();

    /**
     * @brief Returns the current transaction timeout in milliseconds
     *
     * @return int
    */
    int timeout() const { return _timeout;}

    /**
     * @brief Sets the transaction timeout time in milliseconds.
     *
     * @param t
    */
    void setTimeout(int t) { _timeout = t;}
    /**
     * @brief Finds the named packet driver in the registered packet drivers. Returns null if not found.
     *
     * @param s
    */
    static PacketDriver * driver(QString s) { return _drivers[s];}
    /**
     * @brief Processes all registered packet drivers
     *
    */
    static void  processAll();
    /**
     * @brief This is the packet driver's state machine. Returns true if it should be called again,
    typically on a state change.
     *
     * @return bool
    */
    virtual bool process(); // process
    /**
     * @brief Returns true if a message can be sent. By default this requires an open I/O device.
     *
     * @return bool
    */
    virtual bool canSend();

    /**
     * @brief Returns true if the start of packet has been received. If white space skipping is enabled then
     * white space always returns false before trying to match the start character. If the start character is zero all characters return true
     *
     * @param c Current character
     * @return bool
    */
    virtual bool isStart(char c)
    {
        if(_skipWhiteSpace && (c <= ' ')) return false;
        return (_startChar == 0) || (c == _startChar);
    } // hunt the start of the reply - default to non-blank

    /**
     * @brief Determines if a packet has ended.
     * End character must be non-null
     * If there is no end char this must be overloaded
     * @param c
     * @return bool
    */
    virtual bool isEnd(char c)
    {
        return (c != 0) && (c == _endChar);
    }
    /**
     * @brief Sets the QIODevice to read and write on.
     *
     * @param c The QIODevice
    */
    void setIoDevice(QIODevice *c)
    {
        _chan = c;
    } // set the IO channel
    /**
     * @brief Clears the request queue
     *
    */
    virtual void clear() // clear the packet driver
    {
        QMutexLocker l(&_mux);
        _queue.clear();
        _rx.close();
        _rx.buffer().clear();
    }

    // packet framing
    /**
     * @brief Gets the start character. Zero sets no start character
     *
     * @return char
    */
    char startChar() const { return _startChar;}
    /**
     * @brief Gets the packet end character. Zero is no end character
     *
     * @return char
    */
    char endChar() const { return _endChar;}
    /**
     * @brief Sets the start character for the packet. Zero is no start character. The start character is
     * sent before the packet data.
     *
     * @param c
    */
    void setStartChar(char c) { _startChar = c;}
    /**
     * @brief Sets the end character. Zero is no end character. The end character is sent after the packet data.
     * If a zero end character is set the the packetEnd function must be overloaded to detect the end of the response.
     *
     * @param c
    */
    void setEndChar(char c) { _endChar = c;}
    /**
     * @brief If true all white space (characters less than or equal to space) are ignored before
     * testing for a start character.
     *
     * @return bool
    */
    bool skipWhiteSpace() const  { return _skipWhiteSpace;}
    /**
     * @brief  Sets the white space skip flag.
     *
     * @param f
    */
    void setSkipWhiteSpace(bool f) { _skipWhiteSpace = f;}
    //
    /**
     * @brief Returns the number of bytes waiting in any receive buffer
     *
     * @return qint64
    */
    qint64 bytesAvailable ()
    {
        if(_chan) return _chan->bytesAvailable();
        return 0;
    }
    //
    /**
     * @brief fetches as much as possible of the packet - by default reads chars until start then until end found
     * UDP protocols send all data in one packet that must be read in one step
     *
     * @return bool
    */
    virtual bool getPacket();
    /**
     * @brief Returns a reference to the data packet at the head of the queue. That is the request that
     the response is being waited for. If the queue is empty then the receive buffer is returned
     *
     * @return QByteArray &
    */
    QByteArray & head()
    {
        if(_queue.count()) return _queue.head()._packet;
        return _rx.buffer();
    }
    //

    /**
     * @brief Returns a reference to the receive buffer
     *
    */
    QByteArray & rxBuffer() { return _rx.buffer();}
    //
    /**
     * @brief Returns true if all of a packet has been received. Typically this is overloaded to handle protocols that
     include a packet length at the start of the response (eg MODBUS)
     *
     * @return bool
    */
    virtual bool packetEnd() { return false;} // test for packet end after all data read
    //
    /**
     * @brief Sends a reply to the requesting objects and resets the state machine to send the next request,
     * if any.
     *
    */
    virtual void sendResponse();

    /**
     * @brief Retries unless retried PACKET_DRIVER_MAXRETRY time already in which case an error notify is sent
     to caller.
     *
    */
    virtual void errorResponse();
    //
signals:
    /**
     * @brief Emitted when a response has been received or an error condition detected
     *
     * @param caller Caller id
     * @param requestId Request Id
     * @param response The response data
     * @param error True if there was an error, usually a timeout.
    */
    void  haveResponse(unsigned /*caller */, unsigned /*request id*/,
                       QByteArray /*response*/, bool /*error*/);
public slots:
    /**
     * @brief queue a request to the interface. The number of queued requests is limited to PACKET_DRIVER_MAX_QUEUE_LEN.
     *
     *
     * @param c Caller Id
     * @param r Request Id
     * @param p Request data
    */
    virtual void enqueue(unsigned c, unsigned r, QByteArray p);

    /**
     * @brief Clears the receive buffers. Usually called before making a request.
     *
    */
    virtual void flush()
    {
        if(_chan) _chan->readAll(); // flush Rx
    }
};

/**
 * @brief This class is the thread that drives the packet drivers.
 *
*/
class PacketDriverApplication : public QThread
{
    Q_OBJECT
public:
    PacketDriverApplication(QObject *p = 0) : QThread(p){}
    static PacketDriverApplication _global;
    //
    static PacketDriverApplication & global() { return _global;}
    void run();
    //
public slots:
    void tick();
};

#endif // PACKETDRIVER_H
