/*!
 * \file packetdriver.cpp
 *
 * \author B. J. Hill
 * \date __DATE__
 *  License:  GNU LESSER GENERAL PUBLIC LICENSE 2.1
 *  (c)  Micro Research Limited 2010 -
 *  $Id$
 */
#include "packetdriver.h"

// set of packet drivers
QMap<QString,PacketDriver *> PacketDriver::_drivers;

PacketDriverApplication  PacketDriverApplication::_global;

/**
 * @brief
 *
*/
void PacketDriverApplication::run()
{
    QTimer t;
    DBG("Start Timer")
    connect(&t,SIGNAL(timeout()),this,SLOT(tick()));
    t.start(10); // run
    DBG("Execing Thread")
    exec();
}

/**
 * @brief
 *
*/
void PacketDriverApplication::tick()
{
    //DBG("Processing Packets")
    PacketDriver::processAll();
}


/**
 * @brief processAll drivers
 *
*/
void  PacketDriver::processAll()
{
    for(QMap<QString,PacketDriver *>::iterator i = _drivers.begin();
                                    i != _drivers.end(); ++i)
    {
        if(i.value())
            while(i.value()->process());
    }
}


/**
 * @brief Constructs packet driver and registers
 *
 * @param n
 * @param parent
 * @return
*/
PacketDriver::PacketDriver(QString n, QObject *parent) :
    QObject(parent),_packetStart(false),
    _state(0),
    _waiting(true),
    _chan(0),
    _timeout(1000),
    _startChar(0),
    _endChar(0),
    _skipWhiteSpace(true)
{
    _drivers[n] = this;
    setObjectName(n);
    DBG("Name " << n)
}

/**
 * @brief Destructor - unregister
 *
*/
PacketDriver::~PacketDriver()
{
    _drivers.take(objectName());
}



/**
 * @brief
 *
 * @return bool
*/
bool PacketDriver::canSend()
{
    return _chan && _chan->isOpen();
}

void PacketDriver::sendResponse()
{
    unsigned c;
    unsigned r;
    QByteArray b;
    // must avoid recussion locking
    {
        QMutexLocker l(&_mux);
        _rx.close();
        //
        DBG("Rx Response Caller" << _queue.head()._caller << " Request " << _queue.head()._request )
        c = _queue.head()._caller;
        r = _queue.head()._request;
        b = _rx.buffer();
        _packetStart = false;
        _waiting = true;
        _queue.dequeue();
    }
    emit haveResponse(c,r,b,false);
}

bool PacketDriver::getPacket()
{
    bool ret = false;
    char c = 0;
    DBG("Bytes Available " << _chan->bytesAvailable())
    while(_chan->getChar(&c))
    {
        // wait for
        DBG("Got Char " << (int)c)
        if(_packetStart)
        {
            if(!isEnd(c))
            {
                _rx.write(&c,1);
            }
            else
            {
                DBG("Got packet end")
                sendResponse();
                ret = true;
                break;
            }
        }
        else
        {
            if((_packetStart = isStart(c)))
            {
                DBG("Got Start " << (int)c)
                _rx.close();
                _rx.buffer().clear();
                _rx.open(QIODevice::ReadWrite);
                _rx.write(&c,1);
            }
        }
    }
    return ret;
}

void PacketDriver::errorResponse()
{
    unsigned c = 0;
    unsigned r = 0;
    {
        QMutexLocker l(&_mux);
        _packetStart = false;
        _waiting = true;
        if( !_queue.isEmpty() &&  (_queue.head()._retry++ > PACKET_DRIVER_MAXRETRY))
        {
            DBG("Timeout Response " << _queue.head()._caller << " " << _queue.head()._request)
            c = _queue.head()._caller;
            r = _queue.head()._request;
            _queue.dequeue();
        }
    }
    emit haveResponse(c,r,QByteArray(),true);
}

/**
 * @brief process loop - drives the send / receive
 *
 * @return bool
*/
bool PacketDriver::process()
{
    bool ret = false;
    if(canSend()) // is the channel open
    {
        if(_waiting)
        {
            QMutexLocker l(&_mux);
            if(_queue.count())
            {
                // send the packet
                DBG("Send " << byteArrayToString(_queue.head()._packet))
                flush();
                // must write in one step so it works with UDP
                QByteArray b = _queue.head()._packet;
                // frame the packet
                if(_startChar) b.prepend(_startChar);
                if(_endChar) b.append(_endChar);
                //
                _chan->write(b);
                _timer.start();
                _waiting = false;
                ret = true;
            }
        }
        else
        {
            ret = getPacket();
            if(_packetStart && packetEnd())
            {
                // no more data, do we have all of the packet?
                DBG("packetEnd Found")
                sendResponse();
                ret = true;
            }
            else
            {
                ret = false;
            }
            //
            if(_timer.elapsed() > _timeout)
            {
                DBG("Timeout")
                errorResponse();
            }
        }
    }
    return ret;
}

/**
 * @brief
 *
 * @param c
 * @param r
 * @param p
*/
void PacketDriver::enqueue(unsigned c, unsigned r, QByteArray p)
{
    DBG("Enqueue c = " << c << " r = " << r )
    QMutexLocker l(&_mux);
    DBG("Got Lock" )

    if(_queue.count() < PACKET_DRIVER_MAX_QUEUE_LEN)
    {
        Packet pkt(c,r,p);
        _queue.enqueue(pkt);
    }
}

