#ifndef MODBUS_HPP
#define MODBUS_HPP
/*!
 * \file modbus.hpp
 *
 * \author B. J. Hill
 * \date __DATE__
 *  License:  GNU LESSER GENERAL PUBLIC LICENSE 2.1
 *  (c)  Micro Research Limited 2010 -
 *  $Id$
 */

	//
	// Purpose: to provide modbus encapsulation
	// (c) B. J. Hill 2007-
	// $Id: modbus.hpp,v 1.1.1.1 2010/06/08 14:15:59 barry Exp $
	//
#include <Utility/base.hpp>
#include "MrlHardware_global.h"
//
#include <serialpacketdriver.h>
#include <tcppacketdriver.h>
#include <udppacketdriver.h>
//
	//
	// Some lifted from Cal-Controls public domiain code
	//
	// a modbus message
	//
	// max length of modbus packet
	//
#define MAX_MODBUS_LEN 256
#define MODBUS_READ_HOLDING     3
#define MODBUS_WRITE_SINGLE     6
#define MODBUS_READ_COIL        1
#define MODBUS_WRITE_COIL       5
#define MODBUS_READ_INPUT       4
#define MODBUS_WRITE_MULTIPLE_COILS 0x0F
#define MODBUS_WRITE_MULTIPLE_REG 0x10
	//
#define MODBUS_CAL_ENTERBYTE   0x05
#define MODBUS_CAL_EXITBYTE    0x06
	//
/*!
 * \class MODBUSMSG
 *
 * \brief Encapsulates a MODBUS message
 *
 * This class allows MODBUS messages to be built and converted to/from ASCII and MODBUS/TCP.
 * Remember that MODBUS register addresses are documented one based but the messages address registers zero based.
 * This means the address in the message is one less than the address in the documentation.
 *
 * \author B. J. Hill
 *
 */

        class MRLHARDWARESHARED_EXPORT MODBUSMSG
	{
                unsigned char data[MAX_MODBUS_LEN + 6]; // the data - allow for modbus TCP header/**< TODO */

		private:
                unsigned Length;

		public:
		//
                /**
                 * @brief Returns a pointer to the data block
                 *
                */
                unsigned char *getData() { return data;}
                /**
                 * @brief Returns the length of MODBUS message
                 *
                 * @return unsigned
                */
                unsigned getLength() const {return Length;}
                /**
                 * @brief Sets the message length
                 *
                 * @param s Message length
                */
                void setLength(unsigned s) { Length = s;}
		//

                class iterator // read iterator for stepping through messages
                {
                    MODBUSMSG &_m; // the message
                    unsigned int index;
                public:
                    iterator(MODBUSMSG &m ) : _m(m),index(0) {}
                    void start(int i) {index = i;}
                    //
                    unsigned char getByte()
                    {
                        return  _m.getByte(index++);
                    }
                    unsigned int getWord()
                    {
                        unsigned int ret = _m.getWord(index);
                        index += 2;
                        return ret;
                    }
                    unsigned long getLong()
                    {
                        unsigned long ret = getWord() << 16 | getWord() ;
                        return ret;
                    }
                    float getFloat()
                    {
                        float ret = _m.getFloat(index);
                        index += 4;
                        return ret;
                    }
                };

		public:
		//
                /**
                 * @brief Constructs and zeros a MODBUS message
                 *
                */
                MODBUSMSG() ;
                /**
                 * @brief Initialises a MODBUSMSG from a QByteArray
                 *
                 * @param QByteArray
                */
                MODBUSMSG(QByteArray);
		//
                /**
                 * @brief Initialises (zeros) the MODBUSMSG
                 *
                */
                void initialise();
		//
                /**
                 * @brief Adds a byte value to the end of a message and increments the message length
                 *
                 * @param c
                */
                void addByte(unsigned char c);

                /**
                 * @brief Adds a two byte integer value to a message. MSB first. Increments the length by two.
                 *
                 * @param v
                */
                void addWord(unsigned short v);
                /**
                 * @brief Adds a four byte integer value to a message, MSB first. Increments the length by four.
                 *
                 * @param v
                */
                void addLong(unsigned long v);
                /**
                 * @brief Adds a four byte float value to a message, MSB first. Increments the length by four.
                 *
                 * @param v
                */
                void addFloat(float v);

		//
                /**
                 * @brief Reads a byte from a message at the given offset
                 *
                 * @param off
                 * @return unsigned char
                */
                unsigned char getByte (unsigned off);

		//
		// assume little endian
		// so must switch byte order
                /**
                 * @brief Gets the two byte integer, converting to little endian (if required) at the given offset.
                 *
                 * @param off
                 * @return unsigned short
                */
                unsigned short getWord (unsigned off);

                /**
                 * @brief Gets the four byte integer, converting to little endian (if required) at the given offset.
                 *
                 * @param off
                 * @return unsigned long
                */
                unsigned long getLong (unsigned off);

                /**
                 * @brief Gets the four byte float, converting to little endian (if required) at the given offset.
                 *
                 * @param off
                 * @return float
                */
                float getFloat(unsigned off);

		//
                /**
                 * @brief Initialises a message with the address and op code.
                 *
                 * @param node The instrument's address
                 * @param op The MODBUS opcode
                */
                void start(unsigned node, unsigned op);

                /**
                 * @brief Creates a message to write one value to a holding register
                 *
                 * @param node The instrument's address
                 * @param reg Register to read , zero based
                 * @param value Value to write
                */
                void createWriteHolding(unsigned node, unsigned reg, unsigned value);

                /**
                 * @brief Creates a message to write one value to a holding register
                 *
                 * @param node The instrument's address
                 * @param reg Register to read , zero based
                 * @param count number of registers to write
                */
                void createMultipleWriteHolding(unsigned node, unsigned reg, unsigned count);
                /**
                 * @brief Creates a message to write one value to a holding register
                 *
                 * @param node The instrument's address
                 * @param reg Register to read , zero based
                 * @param values to write
                */
                void createMultipleWriteHolding(unsigned node, unsigned reg, QList<unsigned> values);

                /**
                 * @brief Creates a message to read one value from a holding register
                 *
                 * @param node The instrument's address
                 * @param reg Register to read , zero based
                 * @param value  Value to write
                */
                void createReadHolding(unsigned node, unsigned reg,unsigned count = 1);

                /**
                 * @brief Creates a message to read an input register
                 *
                 * @param node The instrument's address
                 * @param reg Register to read , zero based
                 * @param count Number of registers to read
                */
                void createReadInput(unsigned node, unsigned reg,unsigned count = 1);

                /**
                 * @brief
                 *
                 * @param node The instrument's address
                 * @param reg Register to read , zero based
                 * @param value  Value to write
                */
                void createWriteCoil(unsigned node, unsigned reg, bool value);

                /**
                 * @brief
                 *
                 * @param node The instrument's address
                 * @param reg Register to read , zero based
                */
                void createReadCoil(unsigned node, unsigned reg);


		//
		// ASCII LRC calculation on buffer
		//
                /**
                 * @brief Calculates the checksum for a MODBUS ASCII packet
                 *
                 * @return unsigned
                */
                unsigned LRC();
		// CRC for RTU
		//
                /**
                 * @brief Calculates the two byte checksum for an RTU packet
                 *
                 * @return unsigned
                */
                unsigned CRC();
                /**
                 * @brief Returns the instrument address. This is the first byte.
                 *
                 * @return unsigned
                */
                unsigned address() const;
                /**
                 * @brief Returns the opcode for the message. This is the second byte.
                 *
                 * @return unsigned
                */
                unsigned op() const;
		//
                // only call this prior to sending the packet
                /**
                 * @brief Adds the CRC to the message. Adds two to the message length
                 *
                */
                void addCRC();

		//
                /**
                 * @brief Adds the LRC to the message. Adds one to the message length
                 *
                */
                void addLRC();

		//
                /**
                 * @brief Gets the nth register, Accounts for different message header lengths.
                 * Registers are two bytes.
                 *
                 * @param n Register.
                 * @return unsigned
                */
                unsigned getRegister(unsigned n);

		//
		// load RTU packet and verify the CRC
		//
                /**
                 * @brief  Initialises a MODBUSMSG from the given data and length.
                 * Verifies the checksum. Returns true if the checksum validates.
                 *
                 * @param p Message data
                 * @param l  Message length
                 * @return bool
                */
                bool setRtuPacket(const char *p, unsigned l);

		//
		// load a MODBUSTCP packet
		//
                /**
                 * @brief Initialises a MODBUSMSG from the given data
                 * Removes the six byte header.
                 * @param p Message data
                 * @param l
                */
                bool setModbusTCP(const char *p, unsigned l);

		//
		// convert from RTU to MODBUS TCP
		//
                /**
                 * @brief Converts a message to MODBUS/TCP
                 * Sets the message transaction Id. If the id is zero a autoincrementing index is used.
                 * @param transId
                */
                void toModbusTCP(unsigned transId = 0);

		//
		//
                /**
                 * @brief Converts a message to MODBUS ASCII. Adds the LRC.
                 *
                 * @return QByteArray
                */
                QByteArray toAscii(); // assumes the LRC has NOT been set


                /**
                 * @brief Converts a QByteArray with the message in MODBUS ASCII text form into binary.
                 * Returns true if the message validates,
                 * @param b  The ascii message
                 * @return bool
                */
                bool fromAscii(QByteArray b); // must start with a colon and have all white space removed

                /**
                 * @brief Converts the message into a QByteArray.
                 *
                 * @return QByteArray
                */
                QByteArray toByteArray();

                /**
                 * @brief Returns the expected length for a response to a message. Requires the first four bytes
                 *  to have been received.
                 *
                 * @param p A buffer with the received bytes.
                 * @return int
                */
                static int modbusMessageLen(unsigned char *p);

                /**
                 * @brief Gets the coil value from a message.
                 *
                 * @param coil Coil number
                 * @return bool
                */
                bool getCoil(int coil);
                /**
                 * @brief Sets a coil value in a message
                 *
                 * @param coil Coil index
                 * @param value The value to set
                */
                void setCoil(int coil, bool value);

	};

        //
        // packet drivers for MODBUS interfaces
        // RTU
        /*!
         * \class ModbusRtuSerialPacketDriver
         *
         * \brief This packet driver is configured for RTU messages.
         *
         *
         *
         * \author B. J. Hill
         */

        class ModbusRtuSerialPacketDriver
            : public SerialPacketDriver
        {
            bool _haveLast;
        public:
            /**
             * @brief Constructs.
             *
             * @param name Packet driver name
             * @param parent    Parent QObject
            */
            ModbusRtuSerialPacketDriver(QString name ,QObject *parent = 0)
                : SerialPacketDriver(name ,parent)
            {
                setSkipWhiteSpace(false);
            }
            //
            // packet framing
            //
            /**
             * @brief Returns true if the character is the start of message. This must be the address.
             *
             * @param c Character to test.
             * @return bool
            */
            bool isStart(char c)
            {
                return (head()[0] == c); // look for the address byte
            }

            /**
             * @brief Returns true if the end of packet has been recieved.
             *
             * @return bool
            */
            bool packetEnd()
            {
                bool ret = false;
                if(rxBuffer().count() > 4) // need address, op and length
                {
                   unsigned char *p =
                           (unsigned char *)rxBuffer().data();
                   //
                   int l = MODBUSMSG::modbusMessageLen(p);
                   //GDS << "Packet Len " << rxBuffer().size() << endl << byteArrayToString(rxBuffer());
                   if(rxBuffer().count() >= l)
                   {
                        MODBUSMSG m;
                        if(m.setRtuPacket((const char *)p,l)) // validate - CS correct ? - must have correct address byte to get here
                        {
                            //GDS << "Head " << QString::number((unsigned)head().at(1)) << " p[1] " << QString::number((unsigned)p[1]) << " p[2] " << QString::number((unsigned)p[2]);
                          if(head().at(1) == (char)p[1]) ret =  true;
                        }
                        if(!ret)
                        {
                            //GDS << " RTU Error Response " ;
                            errorResponse(); // retry or fail
                        }
                   }
                }
                return ret;
            }
            //
        };
        //
        // MODBUS RTU over TCP is fraught with timing problems
        /*!
         * \class ModbusRtuTcpPacketDriver
         *
         * \brief This packet driver handles RTU over TCP
         *
         *
         *
         * \author B. J. Hill
         * \date
         */

        class ModbusRtuTcpPacketDriver : public TcpPacketDriver
        {
        public:
            /**
             * @brief Constructs.
             *
             * @param name Packet driver name
             * @param parent    Parent QObject
            */
            ModbusRtuTcpPacketDriver
                    (QString host, int port ,QObject *parent = 0)
                : TcpPacketDriver(host, port ,parent)
            {
                setSkipWhiteSpace(false);
            }
            /**
             * @brief Returns true if the character is the start of message. This must be the address.
             *
             * @param c Character to test.
             * @return bool
            */
            bool isStart(char c)
            {
                return (head()[0] == c); // look for the address byte
            }
            /**
             * @brief Returns true if the end of packet has been recieved.
             *
             * @return bool
            */
            bool packetEnd()
            {
                bool ret = false;
                GDS << "Packet End";
                if(rxBuffer().count() > 4) // need address, op and length
                {
                   unsigned char *p =
                           (unsigned char *)rxBuffer().data();
                   //
                   int l = MODBUSMSG::modbusMessageLen(p);
                   //GDS << "Packet Len " << rxBuffer().size() << endl << byteArrayToString(rxBuffer());
                   if(rxBuffer().count() >= l)
                   {
                        MODBUSMSG m;
                        if(m.setRtuPacket((const char *)p,l)) // validate - CS correct ? - must have correct address byte to get here
                        {
                            //GDS << "Head " << QString::number((unsigned)head().at(1)) << " p[1] " << QString::number((unsigned)p[1]);
                          if(head().at(1) == (char)p[1]) ret =  true;
                        }
                        if(!ret)
                        {
                            //GDS << " RTU Error Response " ;
                            errorResponse(); // retry or fail
                        }
                   }
                }
                return ret;
            }
        };
        // UDP packets arrive as a single quantum
        /*!
         * \class ModbusRtuUDPPacketDriver
         *
         * \brief This packet driver handles RTU over UDP
         *
         *
         *
         * \author B. J. Hill
         * \date
         */
        class ModbusRtuUdpPacketDriver : public UdpPacketDriver
        {
        public:
            /**
             * @brief Constructs.
             *
             * @param name Packet driver name
             * @param parent    Parent QObject
            */
            ModbusRtuUdpPacketDriver(QString host, int port,QObject *parent = 0) :
            UdpPacketDriver(host, port,parent)
            {
                setSkipWhiteSpace(false);
            }
            /**
             * @brief Returns true if the character is the start of message. This must be the address.
             *
             * @param c Character to test.
             * @return bool
            */
            bool isStart(char c)
            {
                return (head()[0] == c); // look for the address byte
            }
            /**
             * @brief Returns true if the end of packet has been recieved.
             * UDP will always come across in one lump
             * @return bool
            */
            bool packetEnd()
            {
                bool ret = false;
                if(rxBuffer().count() > 4) // need address, op and length
                {
                   unsigned char *p =
                           (unsigned char *)rxBuffer().data();
                   //
                   int l = MODBUSMSG::modbusMessageLen(p);
                   if(rxBuffer().count() >= l)
                   {
                        MODBUSMSG m;
                        if(m.setRtuPacket((const char *)p,l)) // validate - CS correct ? - must have correct address byte to get here
                        {
                          if(head().at(1) == p[1]) ret =  true;
                        }
                        if(!ret)
                        {
                            errorResponse(); // retry or fail
                        }
                   }
                }
                return ret;
            }
        };

        // MODBUS / TCP - we know the packet length
        // packets should arrive in one go
        /*!
         * \class ModbusTCPPacketDriver
         *
         * \brief This packet driver handles MODBUS/TCP
         TODO - Connect and Disconnect as necessary. Avoid permanent connects
         * \author B. J. Hill
         */
        class ModbusTcpPacketDriver : public TcpPacketDriver
        {
        public:
            /**
             * @brief Constructs.
             *
             * @param host Hostname or IP address
             * @param port Port to connect on, usually 502
             * @param parent Parent QObject
             * @return
            */
            ModbusTcpPacketDriver
                    (QString host, int port ,QObject *parent = 0)
                : TcpPacketDriver(host, port ,parent)
            {
                setSkipWhiteSpace(false);
            }
            /**
             * @brief Returns true if the character is the start of message. This must be the address.
             *
             * @param c Character to test.
             * @return bool
            */
            bool isStart(char )
            {
                if(bytesAvailable() < 9) return false;
                // six byte header plus address, op, 2 bytes of checksum
                return true;
            }
            /**
             * @brief Returns true if the end of packet has been recieved.
             *
             * @return bool
            */
            bool packetEnd()
            {
                // have we got all of the packet
                if(rxBuffer().count() > 5)
                    return (rxBuffer().count() == (rxBuffer().data()[5] + 6));
                return false;
            }
        };

#endif
	
