/*!
 * \file modbus.cpp
 *
 * \author B. J. Hill
 * \date __DATE__
 *  License:  GNU LESSER GENERAL PUBLIC LICENSE 2.1
 *  (c)  Micro Research Limited 2010 -
 *  $Id$
 */

//
// (c) B. J. Hill 2007 -
// Purpose: modbus interface and packet drivers
// $Id: modbus.cpp,v 1.1.1.1 2010/06/08 14:15:58 barry Exp $
//
#include "modbus.hpp"

/**
 * @brief
 *
 * @return
*/
MODBUSMSG::MODBUSMSG() : Length(0)
{
        memset(data,0,sizeof(data));
}

/**
 * @brief
 *
 * @param b
 * @return
*/
MODBUSMSG::MODBUSMSG(QByteArray b)
{
    initialise();
    Length = b.size();
    memcpy(data,(unsigned char *)b.data(),Length);
}

/**
 * @brief
 *
 * @return unsigned
*/
unsigned MODBUSMSG::address() const
{
    return (unsigned)data[0];
}
/**
 * @brief
 *
 * @return unsigned
*/
unsigned MODBUSMSG::op() const
{
    return (unsigned)data[1];
}
//
/**
 * @brief
 *
*/
void MODBUSMSG::initialise()
{
        Length = 0;
        memset(data,0,sizeof(data));
}
//
/**
 * @brief
 *
 * @param c
*/
void MODBUSMSG::addByte(unsigned char c)
{
        if(Length < (MAX_MODBUS_LEN - 1))
        {
                data[Length++] = c;
        };
}
/**
 * @brief
 *
 * @param v
*/
void MODBUSMSG::addWord(unsigned short v)
{
        unsigned char *p = (unsigned char *)&v;
        addByte(p[1]);
        addByte(p[0]);
}
/**
 * @brief
 *
 * @param v
*/
void MODBUSMSG::addLong(unsigned long v)
{
        unsigned char *p = (unsigned char *)&v;
        addByte(p[3]);
        addByte(p[2]);
        addByte(p[1]);
        addByte(p[0]);
}
/**
 * @brief
 *
 * @param v
*/
void MODBUSMSG::addFloat(float v)
{
        unsigned char *p = (unsigned char *)&v;
        addByte(p[3]);
        addByte(p[2]);
        addByte(p[1]);
        addByte(p[0]);
}
//
/**
 * @brief
 *
 * @param off
 * @return unsigned char
*/
unsigned char MODBUSMSG::getByte (unsigned off)
{
        return data[off];
}
//
// assume little endian
// so must switch byte order
/**
 * @brief
 *
 * @param off
 * @return unsigned short
*/
unsigned short MODBUSMSG::getWord (unsigned off)
{
    unsigned short v = getByte(off) << 8 | getByte(off + 1);
    return v;
}
/**
 * @brief
 *
 * @param off
 * @return unsigned long
*/
unsigned long MODBUSMSG::getLong (unsigned off)
{
        unsigned long v = getWord(off) << 16 | getWord(off + 2);
        return v;
}
/**
 * @brief
 *
 * @param off
 * @return float
*/
float MODBUSMSG::getFloat(unsigned off)
{
        union
        {
            float v;
            unsigned long u;
        };
        u = getWord(off) << 16 | getWord(off + 2);
       return v;
}
//
/**
 * @brief
 *
 * @param node
 * @param op
*/
void MODBUSMSG::start(unsigned node, unsigned op)
{
        initialise();
        data[0] = node;
        data[1] = op;
        Length = 2;
}
/**
 * @brief
 *
 * @param node
 * @param reg
 * @param value
*/
void MODBUSMSG::createWriteHolding(unsigned node, unsigned reg, unsigned value)
{
        start(node,MODBUS_WRITE_SINGLE); // initialise packet
        addWord(reg); // register to write to
        addWord(value); // value to write
}

/**
 * @brief
 *
 * @param node
 * @param reg
 * @param count
*/


void MODBUSMSG::createMultipleWriteHolding(unsigned node, unsigned reg, unsigned count)
{
    start(node, MODBUS_WRITE_MULTIPLE_REG);
    addWord(reg);
    addWord(count);
    addByte(count *2);

}

/**
 * @brief
 *
 * @param node
 * @param reg
 * @param values
*/

void MODBUSMSG::createMultipleWriteHolding(unsigned node, unsigned reg, QList<unsigned> values)
{
    start(node, MODBUS_WRITE_MULTIPLE_REG);
    addWord(reg);
    addWord(values.count());
    addByte(values.count() *2);
    for(int i = 0; i < values.count(); i++)
    {
        addWord(values[i]);
    }
}


/**
 * @brief
 *
 * @param node
 * @param reg
 * @param count
*/
void MODBUSMSG::createReadHolding(unsigned node, unsigned reg,unsigned count)
{
        start(node,MODBUS_READ_HOLDING); // initialise packet
        addWord(reg); // register to write to
        addWord(count); // value to write
}
/**
 * @brief
 *
 * @param node
 * @param reg
 * @param count
*/
void MODBUSMSG::createReadInput(unsigned node, unsigned reg,unsigned count)
{
        start(node,MODBUS_READ_INPUT); // initialise packet
        addWord(reg); // register to write to
        addWord(count); // value to write
}

/**
 * @brief
 *
 * @param node
 * @param reg
*/
void MODBUSMSG::createReadCoil(unsigned node, unsigned reg)
{
        start(node,MODBUS_READ_COIL); // initialise packet
        addWord(reg); // register to write to
        addWord(1);
}
/**
 * @brief
 *
 * @param node
 * @param reg
 * @param value
*/
void MODBUSMSG::createWriteCoil(unsigned node, unsigned reg, bool value)
{
        start(node,MODBUS_WRITE_SINGLE); // initialise packet
        addWord(reg); // register to write to
        addWord(value?0x00FF:0); // value to write
}
//
// ASCII LRC calculation on buffer
//
/**
 * @brief
 *
 * @return unsigned
*/
unsigned MODBUSMSG::LRC()
{
        int cs = 0;
        unsigned char *p = data;
        for(unsigned i = 0; i < Length; i++)cs += *(p++);
        return (-cs) & 0xFF;
}
//
// CRC for RTU
//
/**
 * @brief
 *
 * @return unsigned
*/
unsigned MODBUSMSG::CRC()
{
        unsigned short    CRCValue;
        // +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
        // +    Reset the CRC value
        // +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
        CRCValue=0xffff;
        // +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
        // +    Loop for all bytes in message
        // +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
        for (unsigned f=0;f< Length;f++)
        {
                // +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
                // +    Mask the byte value onto the CRC value
                // +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
                CRCValue=(unsigned short)(CRCValue^(unsigned short)data[f]);
                // +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
                // +    Pass each bit in this byte through the CRC generator
                // +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
                for (unsigned i=0;i<8;i++)
                {
                        // +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
                        // +    If bit zero is set then apply the XOR back onto
                        // +    the shift CRC register
                        // +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
                        if ( (CRCValue&0x01) )
                        {
                                CRCValue=(unsigned short)((CRCValue>>1)^0xa001);
                        }
                        else
                        {
                                CRCValue=(unsigned short)(CRCValue>>1);
                        }
                }
        }
        // +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
        // +    Save the CRC value at the end of the message
        // +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
        return CRCValue;
}
//
// only call this prior to sending the packet
/**
 * @brief
 *
*/
void MODBUSMSG::addCRC()
{
    unsigned v = CRC();
    unsigned char *p = (unsigned char *)&v;
    addByte(p[0]);
    addByte(p[1]);
}
//
/**
 * @brief
 *
*/
void MODBUSMSG::addLRC()
{
        addByte(LRC());
}
//
/**
 * @brief
 *
 * @param n
 * @return unsigned
*/
unsigned MODBUSMSG::getRegister(unsigned n)
{
        switch(op())
        {
                case MODBUS_READ_INPUT:
                case MODBUS_READ_HOLDING:
                return getWord(n*2 + 3);
                default:
                return getWord((n + 1)*2);
        };
        return 0;
}
//
// load RTU packet and verify the CRC
//
/**
 * @brief
 *
 * @param p
 * @param l
 * @return bool
*/
bool MODBUSMSG::setRtuPacket(const char *p, unsigned l)
{
        bool res = false;
        if((l < MAX_MODBUS_LEN) && (l > 4))
        {
                memcpy(data,p,l);
                Length = l - 2;
                res = (CRC() == *((unsigned *)(data + Length)));
        };
        return res;
}
//
// load a MODBUSTCP packet
//
/**
 * @brief Convert MODBUS TCP to RTU then to MODBUSMSG
 *
 * @param p
 * @param l
*/
bool MODBUSMSG::setModbusTCP(const char *p, unsigned l)
{
        if((l < MAX_MODBUS_LEN) && (l > 8))
        {
            memcpy(data,p + 6, l - 6);
            Length = l - 6;
            return true;
        };
        return false;
}
//
// convert from RTU to MODBUS TCP
//
/**
 * @brief
 *
 * @param transId
*/
void MODBUSMSG::toModbusTCP(unsigned transId )
{
        char b[MAX_MODBUS_LEN];
        //
        memcpy(b,data,sizeof(b));
        memset(data,0,sizeof(data));
        //
        // add the session id
        data[5] = Length;
        memcpy(data + 6, b,Length);
        //
        Length += 6;
        //
        data[1] = transId & 0xFF;
        data[0] = (transId >> 8) & 0xFF; // add the transaction ID
        //
}
//
//
/**
 * @brief
 *
 * @return QByteArray
*/
QByteArray MODBUSMSG::toAscii() // assumes the LRC has NOT been set
{
    // NB no leading colon or trailing CR
        QByteArray b;
        QBuffer bf(&b);
        if(bf.open(QIODevice::WriteOnly))
        {
                QTextStream os(&bf);
                addLRC(); // add the checksum
                for(unsigned i = 0; i < Length;i++)
                {
                        QString v;
                        v.sprintf("%02X",(unsigned)data[i]);
                        os << v;
                };
                bf.close();
        };
        return b;
}
// Ascii to hex
/**
 * @brief
 *
 * @param c
 * @return unsigned
*/
static unsigned a2b(char c)
{
        if(c > '9')
        {
                return (c + 9) & 0xF;
        };
        return c & 0xF;
}
/**
 * @brief
 *
 * @param b
 * @return bool
*/
bool MODBUSMSG::fromAscii(QByteArray b) // must start with a colon and have all white space removed
{
        //DBG("Received " << b.data())
        initialise();
        char *p = b.data();
        Length = 0;
        if(*p == ':')
        {
                p++;
                for(int i = 0; (i < b.count()) &&  (*p > ' '); i++ )
                {
                        int hi = a2b(*p);
                        p++;
                        int lo = a2b(*p);
                        p++;
                        data[i] = (hi << 4 ) | lo;
                        Length++;
                        //DBG(" d[" << i << " ] " << QString::number(data[i],16))
                };
        };
        //
        Length--;
        //DBG("Length " << Length << " LRC " << LRC() << " Record CS " << data[Length])
        if(Length > 3)
        {
                return (LRC() == data[Length]);
        };
        return false;
}

/**
 * @brief
 *
 * @return QByteArray
*/
QByteArray MODBUSMSG::toByteArray()
{
    QByteArray b((const char *)data,Length);
    return b;
}

// address + op code + 2 byte checksum
#define MODBUS_BASE_LEN (4)
/**
 * @brief Calculate the packet length given the first 4 bytes
 * of the packet have been received
 *
 * @param p
 * @return int
*/
int MODBUSMSG::modbusMessageLen(unsigned char *p)
{
    int len = MODBUS_BASE_LEN;
    if(p[1] & 0x80)
    {
        len += 1;
    }
    else
    {
        switch(p[1])
        {
        case 1:
        {
            len += (1 + p[2]);
        }
        break;
        case 2:
        {
            len += (1 + p[2]);
        }
        break;
        case 3:
        {
            len += (1 + p[2]);
        }
        break;
        case 4:
            {
                len += (1 + p[2] );
            }
            break;
        case 5:
        case 6:
            {
                len += 4;
            }
            break;
        case 7:
            {
                len += 1;
            }
            break;
        case 8:
            {
                len += 2;
            }
            break;
        case 0x0B:
            {
                len += 4;
            }
            break;
        case 0x0C:
            {
                len += 1 +  p[2];
            }
            break;
        case 0x0F:
            {
                len += 4;
            }
            break;
        case 0x10:
            {
                len += 4;
            }
            break;
        case 0x11:
            {
                len += 1 + p[2];
            }
            break;
        case 0x14:
        case 0x15:
            {
                len += 1 + p[2];
            }
            break;
        case 0x16:
            {
                len += 6;
            }
            break;
        case 0x17:
            {
                len += 1 + p[2];
            }
            break;
        case 0x18:
            {
                len += 4 + p[3] + p[2] * 256;
            }
            break;
        default:
            break;
        }
    }
    DBG("Expected Packet Length = " << len)
    return len;
}

/**
 * @brief
 *
 * @param coil
 * @return bool
*/
bool MODBUSMSG::getCoil(int coil)
{
    bool ret = false;
    // assumes this is the response to read coil
    switch(op())
    {
    case MODBUS_READ_COIL:
        {
            // data 2 has the number of bytes in the record
            int off = coil / 8;
            if( off < data[2]) // in range
            {
                int bit = coil % 8; // which bit
                ret = ((data[off + 3] >> bit) & 1);
            }
        }
        break;
    case MODBUS_WRITE_MULTIPLE_COILS:
        {
            int off = coil / 8;
            if( off < data[6]) // in range
            {
                int bit = coil % 8; // which bit
                ret = ((data[off + 7] >> bit) & 1);
            }
        }
        break;
    default:
        break;
    }
    return ret;
}

/**
 * @brief
 *
 * @param coil
 * @param value
*/
void MODBUSMSG::setCoil(int coil, bool value)
{
    // only meaningful for MODBUS_WRITE_MULTIPLE_COILS
    if(op() == MODBUS_WRITE_MULTIPLE_COILS)
    {
        int off = coil / 8;
        if( off < (MAX_MODBUS_LEN - 7)) // in range
        {
            int bit = coil % 8; // which bit
            if( value)
            {
                // or in
                data[off + 7] |= (1 << bit);
            }
            else
            {
                // mask out
                data[off + 7] &= ~(1 << bit);
            }
        }
    }
}


