#ifndef HTTPCLIENT_H
#define HTTPCLIENT_H
/*!
 * \file httpclient.h
 *
 * \author B. J. Hill
 * \date __DATE__
 *  License:  GNU LESSER GENERAL PUBLIC LICENSE 2.1
 *  (c)  Micro Research Limited 2010 -
 *  $Id$
 */

#include <Utility/base.hpp>
#include <QtNetwork>
#include <HttpServer/HttpServer_global.h>
#include <HttpServer/xrmethodcall.h>
#include <HttpServer/xrmethodresponse.h>
#include <HttpServer/udpclient.h>
#include <HttpServer/Broadcast.h>
#include <Utility/xmltraversal.h>
//
// this is base class for the client side of the http server
//  we expect one per application as each  can handle mutiple services
// but there is no constraint
//
/**
 * @brief This class encapsulates HTTP transactions including XMLRPC. Both blocking and
 event driven version of get,put, post and xmlrpc are supported
 *
*/
class HTTPSERVERSHARED_EXPORT HttpClient : public QNetworkAccessManager
{
	Q_OBJECT
        /**
         * @brief
         *
         * @param QMetaObject::Call
         * @param int
         * @param
         * @return int
        */
        public:
        /**
         * @brief
         *
         * @param 0
        */
        HttpClient(QObject *p = 0);
        //
        /**
         * @brief
         *
         * @param host
         * @param data
         * @param port
         * @return QUrl
        */
        static QUrl getHttpUrl(QString host, QString data, int port) // build the URL for the given host
        {
            QUrl u(QString("http://%1/%2").arg(host).arg(data));
            u.setPort(port);
            return u;
        }
	//
	// call an XML RPC proceedure - does POST
        /**
         * @brief
         *
         * @param host
         * @param method
         * @param args
         * @return QNetworkReply *
        */
        QNetworkReply * xmlRpcCall(QUrl host, QString method, QList<QVariant> args);
	//
	// These are blocking actions  - the reply object will be marked deleteLater
	// - so must be used immediately
	//
        /**
         * @brief
         *
         * @param host
         * @param 10000
         * @return QNetworkReply *
        */
        QNetworkReply * getBlock(QUrl host, int to = 10000); // blocking get
        /**
         * @brief
         *
         * @param host
         * @param data
         * @param 10000
         * @return QNetworkReply *
        */
        QNetworkReply * putBlock(QUrl host, QByteArray data, int to = 10000); // blocking put
        /**
         * @brief
         *
         * @param host
         * @param data
         * @param 10000
         * @return QNetworkReply *
        */
        QNetworkReply * postBlock(QUrl host, QByteArray data, int to = 10000); // blocking post
	// blocking RPC call
        /**
         * @brief
         *
         * @param host
         * @param in
         * @param out
         * @param 10000
         * @return bool
        */
        bool xmlRpcCallBlock(QUrl host, const XRMethodCall &in,
                            XRMethodResponse & out, int to = 10000);
	//
            // These are helpers for ls. mkdir, del and rmdir
            // the implimentation at the server side may not be a file system,
            // it could be a database or an XML document
            // list a directory  - put document into XMLTraversal
            /**
             * @brief
             *
             * @param host
             * @param nameFilter
             * @return QDomDocument
            */
            QDomDocument ls(QUrl host,  QString nameFilter)
            {
                QDomDocument ret;
                host.addQueryItem("ls",nameFilter);
                QNetworkReply * r = postBlock(host,QByteArray());
                if(r && (r->error() == QNetworkReply::NoError))
                {
                    //XMLTraversal d;
                    ret.setContent(r);
                }
                 return ret;
            }

           // create a directory (path)
           /**
            * @brief
            *
            * @param path
            * @return bool
           */
           bool mkdir(QUrl path)
           {
               bool ret = false;
               path.addQueryItem("mkdir","");
                QNetworkReply * r = postBlock(path,QByteArray());
                if(r && (r->error() == QNetworkReply::NoError))
                {
                    ret = true;
                }
               return ret;
           }
           // remove a directory
           /**
            * @brief
            *
            * @param path
            * @return bool
           */
           bool rmdir(QUrl path)
           {
               bool ret = false;
               path.addQueryItem("rmdir","");
                QNetworkReply * r = postBlock(path,QByteArray());
                if(r && (r->error() == QNetworkReply::NoError))
                {
                    ret = true;
                }
               return ret;
           }
           // delete a file
            /**
             * @brief
             *
             * @param path
             * @return bool
            */
            bool del(QUrl path)
           {
               bool ret = false;
               path.addQueryItem("del","");
                QNetworkReply * r = postBlock(path,QByteArray());
                if(r && (r->error() == QNetworkReply::NoError))
                {
                    ret = true;
                }
               return ret;
           }
           //
};
#endif // HTTPCLIENT_H


